using System;
using System.Collections;
using System.Data;
using System.Reflection;
using System.Text;
using gov.va.med.vbecs.DAL.VAL;
using ISSUEUNITCOMMENT = gov.va.med.vbecs.Common.VbecsTables.IssuedUnitComment;

namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// This is the class responsible for determining which units are eligible for issue.
	/// 
	/// README if changing code!
	/// If the level of security required to do something is increasing, the UnitsAvailableForLevelTree
	/// method must be revisited.  For example, if a certain function is not going to require that a 
	/// level 4 is needed instead of a level 3, this function will have to change.  Currently, level 3 
	/// access can execute any overrides in this use case.  So, this function bipasses the user portion
	/// of the override checking.  For more information, please see CR 2081.
	/// 
	/// </summary>
	/// 
	public class PatientUnitIssue : BaseBusinessObject
	{
		#region variables
		
		/// <summary>
		/// Collection class that holds issued units
		/// </summary>
		private IssuedUnits _issuedUnits;
		/// <summary>
		/// Patient that units are being issued for
		/// </summary>
		private BOL.Patient _patient;
		/// <summary>
		/// Current user
		/// </summary>
		private BOL.VbecsUser _user;
		/// <summary>
		/// User list to populate combo box
		/// </summary>
		private SortedList _users;
		/// <summary>
		/// Ordered units for the patient (previously selected in UC3 (Select units for patient)) 
		/// </summary>
		private DataTable _dtUnits;
        
        /// <summary>
        /// CR3260
        /// Ordered units for the patient in the form of an ArrayList (previously selected in 
        /// UC3 (Select units for patient)). Some of the methods that I call require an ArrayList.   
        /// </summary>
        private ArrayList _allUnits;
        
		/// <summary>
		/// Units that can be issued normally
		/// </summary>
		private ArrayList _units;
		/// <summary>
		/// Units that must be issued via emergency issue
		/// </summary>
		private ArrayList _emergencyUnits;
		/// <summary>
		/// Indicates whether or not facility is full service
		/// </summary>
		private bool _isFullService;
		/// <summary>
		/// The GUI sets this property to inform PUI that it has overridden all warnings
		/// on a chosen unit.
		/// </summary>
		private bool _unitOK;
		/// <summary>
		/// Temporarily holds comments related to a unit that will be included in an 
		/// exception report.
		/// </summary>
		private ArrayList _comments;
		/// <summary>
		/// Hospital locations from VistALink
		/// </summary>
		private DataTable _dtLocations;
		/// <summary>
		/// If none of the selected units are valid, this stores the message that is
		/// displayed when the OK button is hit.
		/// </summary>
		private StringBuilder _allUnitInvalidMessage;
		/// <summary>
		/// Ordered units that will be released.
		/// </summary>
		private DataTable _dtReleaseUnits;
		/// <summary>
		/// Indicates whether there are actually units eligible for issue, override or not.
		/// This is used for UI processing.
		/// </summary>
		private bool _unitsAvailable;
		/// <summary>
		/// Exception Report entries
		/// </summary>
		private ArrayList _issueExceptions;
		/// <summary>
		/// Keeps count of the number of autologous units that are available for issue
		/// </summary>
		private int _autologousUnits;
		/// <summary>
		/// Keeps count of the number of directed units that are available for issue
		/// </summary>
		private int _directedUnits;
		/// <summary>
		/// Keeps count of the number of restricted units that are available for issue
		/// </summary>
		private int _restrictedUnits;
		/// <summary>
		/// This bool is only for code that bipasses the security check.
		/// </summary>
		private bool _bipassSecurityLevelCheck;
        /// <summary>
        /// Date/Time when form was opened (CR 3550)
        /// </summary>
        private DateTime _currentDateTime;

		// CR2899 start: cached information for faster processing
		// NOTE: DO NOT access these directly.  Access them through the corresponding
		// property!!!  Or else!
		private Hashtable _specimensCache = null;
		private Hashtable _TASSpecimenResultsCache = null;
		private Hashtable _specimenABORhTestResultsInconsistentCache = null;
		private Hashtable _systemABORhCache = null;
		private Hashtable _componentRequirementsCache = null;
		private Hashtable _antNegCache = null;
		private Common.ABORhSource _patientABORh = Common.ABORhSource.None;
		// CR2899 end


		#region Issued units related - BR_73.24

		// The variables in this region could have been put in the IssuedUnit object instead
		// of here.  Since these properties are identical for every unit in an issue, putting
		// them here makes a little more sense.  See BR_73.24 - C.Jensen, 2/13/2004

		/// <summary>
		/// Physician ordering the transfusion
		/// </summary>
		private string _orderingPhysician;
		/// <summary>
		/// Issue date of the units - must NOT be in the future - REQUIRED
		/// </summary>
		private DateTime _issueDate = VBECSDateTime.GetDivisionCurrentDateTime();
		/// <summary>
		/// This is the name of the issuer - REQUIRED
		/// </summary>
		private string _issuedBy;
		/// <summary>
		/// This is the name or identifier of the transporter - REQUIRED
		/// </summary>
		private string _transporter;
		/// <summary>
		/// Issue To location from VistALink - REQUIRED
		/// </summary>
		private string _issueToLocation;
		/// <summary>
		/// Identification entry number for the location
		/// </summary>
		private int _issueToLocationIEN;
		/// <summary>
		/// Boolean indicating whether or not unit put into remote storage
		/// </summary>
		private bool _remoteStorageIndicator;
		/// <summary>
		/// Identifier for the remote storage location
		/// </summary>
		private string _remoteStorageIdentity;

 
		#endregion

		/// <summary>
		/// Collection of Hashtables.  The key for this is a BloodUnitGuid.
		/// The value is another Hashtable which holds all of the warning messages for
		/// a unit (key) and whether or not the unit can be issued (value).  
		/// 
		/// The point of this is to keep all of the messages in memory so constant trips 
		/// to the DB are prevented.
		/// </summary>
		Hashtable _unitWarnings;

		
		#region EventMessage

		/// <summary>
		/// Issues messages
		/// </summary>
		public delegate void IssueMessagesEventHandler(object sender, IssueMessagesEventArgs e);

		/// <summary>
		/// Event that is raised when messages are generated
		/// </summary>
		public event IssueMessagesEventHandler IssueMessagesEvent;

		/// <summary>
		/// Event that is raised when messages related to a unit's issue are generated
		/// </summary>
		protected virtual void IssueMessagesGenerated(Hashtable messages, BOL.BloodUnit bloodUnit)
		{
			IssueMessagesEventArgs eventArgs = new IssueMessagesEventArgs(messages, bloodUnit);
	
			if (IssueMessagesEvent != null)
				IssueMessagesEvent(this, eventArgs);
		}

		#endregion

		#endregion


		#region constructors

		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3812"> 
		///		<ExpectedInput>Called with no broken rule context</ExpectedInput>
		///		<ExpectedOutput>Broken rule count equal to 0</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3813"> 
		///		<ExpectedInput>Called with no broken rule context of both forms</ExpectedInput>
		///		<ExpectedOutput>Broken rule count equal to 2 or 1 depending on form</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PatientUnitIssue's main and only constructor
		/// BR_73.13, BR_73.17
		/// </summary>
		/// <param name="patient"></param>
		/// <param name="user"></param>
		public PatientUnitIssue(BOL.Patient patient, BOL.VbecsUser user)
		{
            _currentDateTime = DateTime.Now;

			_bipassSecurityLevelCheck = false;
			_patient = patient;
			_user = user;

			_units = new ArrayList();
			_unitWarnings = new Hashtable();
			_allUnitInvalidMessage = new StringBuilder();
			_emergencyUnits = new ArrayList();
			_comments = new ArrayList();
			_isFullService = new Division(Common.LogonUser.LogonUserDivisionCode).ServiceTypeIndicator;

			_users = this.GetUsersSortedList();

			// Inner class (collection) used to hold selected units
			_issuedUnits = new IssuedUnits(_issueDate);

            // BR_73.06
            _dtUnits = BOL.OrderedUnit.GetOrderedUnitsForPatient(_patient.PatientGuid);

            _issueExceptions = new ArrayList();
			
			SortUnits();
		}
		
		/// <summary>
		/// This is a private constructor and must never be made public.  This constructor is
		/// called ONLY to support CR 2081.  DO NOT MAKE PUBLIC! 
		/// 
		/// This CR requires a level 3 user.  So, one has to be simulated.  Unofortunately, due
		/// to the design of the system, it is not possible to just instantiate a new VBECS user
		/// and assign level 3 access.
		/// </summary>
		/// <param name="patient"></param>
		private PatientUnitIssue(BOL.Patient patient)
		{
            _currentDateTime = DateTime.Now;

			_bipassSecurityLevelCheck = true;
			_patient = patient;

			_units = new ArrayList();
			_unitWarnings = new Hashtable();
			_allUnitInvalidMessage = new StringBuilder();
			_emergencyUnits = new ArrayList();
			_comments = new ArrayList();
			_isFullService = new Division(Common.LogonUser.LogonUserDivisionCode).ServiceTypeIndicator;

			_users = this.GetUsersSortedList();

			// Inner class (collection) used to hold selected units
			_issuedUnits = new IssuedUnits(_issueDate);

			_dtUnits = BOL.OrderedUnit.GetOrderedUnitsForPatient(_patient.PatientGuid);
			_issueExceptions = new ArrayList();
			
			SortUnits();
		}

		#endregion


		#region EventArgs class
		
		/// <summary>
		/// This event holds messages related to the issue of units.
		/// </summary>
		public class IssueMessagesEventArgs : System.EventArgs 
		{
			private Hashtable _issueMessages;
			private BOL.BloodUnit _bloodUnit;

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/13/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7573"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7574"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// IssueMessagesEventArgs
			/// </summary>
			/// <param name="issueMessages"></param>
			/// <param name="bloodUnit"></param>
			public IssueMessagesEventArgs(Hashtable issueMessages, BOL.BloodUnit bloodUnit) 
			{
				_issueMessages = issueMessages;
				_bloodUnit = bloodUnit;
			}

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/13/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7575"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7576"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// Messages
			/// </summary>
			public Hashtable Messages
			{
				get
				{
					return _issueMessages;
				}
			}

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/13/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7577"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7578"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// BloodUnit
			/// </summary>
			public BOL.BloodUnit BloodUnit
			{
				get
				{
					return _bloodUnit;
				}
			}
		}
		
		#endregion


		#region properties

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5984"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Patient</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5985"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Patient that units are being issued for.
		/// </summary>
		public BOL.Patient Patient
		{
			get
			{
				return _patient;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5986"> 
		///		<ExpectedInput>string</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5987"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Physisian that ordered the units
		/// </summary>
		public string OrderingPhysician
		{
			get
			{
				return _orderingPhysician;
			}
			set
			{
				_orderingPhysician = value;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3794"> 
		///		<ExpectedInput>Current date and date in past</ExpectedInput>
		///		<ExpectedOutput>No broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3795"> 
		///		<ExpectedInput>Future date</ExpectedInput>
		///		<ExpectedOutput>Broken rule count incremented by 1</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Date unit(s) were issues
		/// BR_73.08
		/// </summary>
		public DateTime IssueDate
		{
			get
			{
				return _issueDate;
			}
			set
			{
				_issueDate = value;

				// The msessages have already been set, but unit and specimen expiration
				// may no longer apply.

				IDictionaryEnumerator enumerator = _unitWarnings.GetEnumerator();
				while ( enumerator.MoveNext() )
				{
					Guid unitGuid = (Guid) enumerator.Key;
					
					Hashtable htUnits = (Hashtable) enumerator.Value;

					// unit expiration
					if (htUnits.Contains(Common.StrRes.ConfMsg.UC073.UnitExpired()) ||
						htUnits.Contains(Common.StrRes.InfoMsg.UC073.UnitExpired()) )
					{
						BOL.BloodUnit bloodUnit = new BOL.BloodUnit(unitGuid);
						if (bloodUnit.BloodUnitMedia.UnitExpirationDate > _issueDate)
						{
							htUnits.Remove(Common.StrRes.ConfMsg.UC073.UnitExpired());
							htUnits.Remove(Common.StrRes.InfoMsg.UC073.UnitExpired());
						}
					}

					BOL.Specimen specimen = this.GetSpecimenAssociatedWithUnit(unitGuid);
					// specimen expiration
					if (specimen!=null && 
						(htUnits.Contains(Common.StrRes.ConfMsg.UC073.SpecimenExpired()) ||
						htUnits.Contains(Common.StrRes.InfoMsg.UC073.SpecimenExpired())) )
					{
						if (specimen.SpecimenExpirationDate > _issueDate)
						{
							htUnits.Remove(Common.StrRes.ConfMsg.UC073.SpecimenExpired());
							htUnits.Remove(Common.StrRes.InfoMsg.UC073.SpecimenExpired());
						}
					}
				}


				// BR_73.08
				RuleBroken(MethodBase.GetCurrentMethod().Name, _issueDate > VBECSDateTime.GetDivisionCurrentDateTime());
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3796"> 
		///		<ExpectedInput>Valid user</ExpectedInput>
		///		<ExpectedOutput>No broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3797"> 
		///		<ExpectedInput>Empty string</ExpectedInput>
		///		<ExpectedOutput>Broken rule count incremented by 1</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="3882"> 
		///		<ExpectedInput>null</ExpectedInput>
		///		<ExpectedOutput>Broken rule count incremented by 1</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Issuing technician
		/// </summary>
		public string IssuedBy
		{
			get
			{
				return _issuedBy;
			}
			set
			{ 	
				if (value == null)
				{
					throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument(MethodBase.GetCurrentMethod().Name).ResString);
				}

				if( _users.ContainsKey( value ) ) 
					_issuedBy = ((BOL.VbecsUser)_users[ value ] ).UserNTLoginID;

				RuleBroken(MethodBase.GetCurrentMethod().Name, _issuedBy == null || _issuedBy.Trim().Length == 0);
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/21/2004</CreationDate>
		///<TestCases>
		///<Case type="0" testid ="5988"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Empty string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5989"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// This is the message that is returned when the patient has units
		/// selected, but none are actually valid for issue.
		/// </summary>
		public string AllUnitsInvalidMessage
		{
			get
			{
				return _allUnitInvalidMessage.ToString();
			}
		}

		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3798"> 
		///		<ExpectedInput>String with length > 0</ExpectedInput>
		///		<ExpectedOutput>No broken rules</ExpectedOutput>
		///	</Case>
		///	
		///	<Case type="0" testid ="3799"> 
		///		<ExpectedInput>null</ExpectedInput>
		///		<ExpectedOutput>Broken rule count incremented by 1</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3811"> 
		///		<ExpectedInput>Empty string</ExpectedInput>
		///		<ExpectedOutput>Broken rule count incremented by 1</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Name of individual transporting the unit(s)
		/// BR_73.09
		/// </summary>
		public string Transporter
		{
			get
			{
				return _transporter;
			}
			set
			{
				_transporter = value;
				// BR_73.09
				RuleBroken(MethodBase.GetCurrentMethod().Name, _transporter == null || _transporter.Trim().Length == 0);
			}
		}

		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3800"> 
		///		<ExpectedInput>Valid string</ExpectedInput>
		///		<ExpectedOutput>No broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3809"> 
		///		<ExpectedInput>Empty string</ExpectedInput>
		///		<ExpectedOutput>Broken rule count incremented by 1</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Location that units are being issued to.
		/// BR_73.09
		/// </summary>
		public string IssueToLocation
		{
			get
			{
				return _issueToLocation;
			}
			set
			{
				_issueToLocation = value;
				// BR_73.09
				RuleBroken(MethodBase.GetCurrentMethod().Name, _issueToLocation == null || _issueToLocation.Trim().Length == 0);
								
				// Replaces ' with '' so query will work correctly
				string filter = _issueToLocation;
				int quoteLocation = filter.IndexOf("'");
				if (quoteLocation > -1)
				{
					filter = filter.Insert(quoteLocation, "'");
				}

				if (_dtLocations!=null)
				{
					DataRow [] dtRowArray = _dtLocations.Select(Common.DatabaseConstants.VistALinkColumnNames.LocationName + " = '" + filter + "'");
					if (dtRowArray.Length == 1)
					{
						_issueToLocationIEN = (int) Convert.ToInt32( dtRowArray[0][Common.DatabaseConstants.VistALinkColumnNames.LocationIen].ToString() );
					}
				}
			}
		}

		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3802"> 
		///		<ExpectedInput>false</ExpectedInput>
		///		<ExpectedOutput>No broken rules</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3803"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>Broken rule count incremented by 1</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Indicates whether or not a remote storage location was specified
		/// BR_73.10
		/// </summary>
		public bool RemoteStorageIndicator
		{
			get
			{
				return _remoteStorageIndicator;
			}
			set
			{
				_remoteStorageIndicator = value;
				RuleBroken("RemoteStorageIdentity", value == true && (_remoteStorageIdentity == null || _remoteStorageIdentity.Trim().Length == 0) );
			}
		}
	
		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3804"> 
		///		<ExpectedInput>String with length > 0</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3805"> 
		///		<ExpectedInput>null value</ExpectedInput>
		///		<ExpectedOutput>Broken rule count incremented by 1</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3921"> 
		///		<ExpectedInput>Empty string</ExpectedInput>
		///		<ExpectedOutput>Broken rule count incremented by 1</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// RemoteStorageIdentity
		/// </summary>
		public string RemoteStorageIdentity
		{
			get
			{
				return _remoteStorageIdentity;
			}
			set
			{
				_remoteStorageIdentity = value;
				RuleBroken(MethodBase.GetCurrentMethod().Name, (_remoteStorageIdentity == null || _remoteStorageIdentity.Trim().Length == 0) && _remoteStorageIndicator == true);
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3923"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Empty ArrayList</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="513"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This arraylist will be used by the UI layer to populate the normal issue tab.
		/// </summary>
		public ArrayList Units
		{
			get
			{
				return _units;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5990"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Empty ArrayList</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5991"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This arraylist will be used by the UI layer to populate the emergency issue tab.
		/// </summary>
		public ArrayList EmergencyUnits
		{
			get
			{
				return _emergencyUnits;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5992"> 
		///		<ExpectedInput>bool value</ExpectedInput>
		///		<ExpectedOutput>none</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="5993"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Allows the GUI layer to inform the BOL whether or not all warnings have been overridden.
		/// </summary>
		public bool UnitOK
		{
			set
			{
				_unitOK = value;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7218"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7219"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Exception 
		/// </summary>
		public ArrayList IssueExceptions
		{
			get
			{
				return _issueExceptions;
			}
		}

		#endregion


		#region Cache Properties

		/// <summary>
		/// 
		/// </summary>
		public Hashtable SpecimensCache
		{
			get
			{
				if 	(_specimensCache  == null)
				{
					_specimensCache  = new Hashtable();
				}
				return _specimensCache;
			}
		}


		private Hashtable TASSpecimenResultsCache
		{
			get
			{
				if 	(_TASSpecimenResultsCache  == null)
				{
					_TASSpecimenResultsCache  = new Hashtable();
				}
				return _TASSpecimenResultsCache;
			}
		}


		private Hashtable SpecimenABORhTestResultsInconsistentCache
		{
			get
			{
				if 	(_specimenABORhTestResultsInconsistentCache  == null)
				{
					_specimenABORhTestResultsInconsistentCache  = new Hashtable();
				}
				return _specimenABORhTestResultsInconsistentCache;
			}
		}


		private Hashtable SystemABORhCache
		{
			get
			{
				if 	(_systemABORhCache  == null)
				{
					_systemABORhCache  = new Hashtable();
				}
				return _systemABORhCache;
			}
		}


		private Hashtable ComponentRequirementsCache
		{
			get
			{
				if 	(_componentRequirementsCache  == null)
				{
					_componentRequirementsCache  = new Hashtable();
				}
				return _componentRequirementsCache;
			}
		}


		private Hashtable AntNegCache
		{
			get
			{
				if 	(_antNegCache  == null)
				{
					_antNegCache  = new Hashtable();
				}
				return _antNegCache;
			}
		}


		

		private Common.ABORhSource PatientABORh
		{
			get
			{
				return _patientABORh;
			}
		}



		#endregion


		#region methods


        ///<Developers>
        ///	<Developer>Carl Jensen</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>5/12/2015</CreationDate>
        ///
        /// <summary>
        /// Returns a boolean that indicates if the ABO/Rh changed in the course of working in Issue Unit
        /// CR 3550
        /// </summary>
        public bool DidABORhChange()
        {
            //Get patient's ABO/Rh testing results
            DataTable dtpatientAboRhHistory = _patient.GetPatientAboRHHistory();
            DataRow[] aboHistory = dtpatientAboRhHistory.Select("(" + Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = 18 OR " +
                                                                  Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = 65 OR " +
                                                                  Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = 118) AND " +
                                                                  Common.VbecsTables.SpecimenTest.TestResultId + " <> '' AND " +
                                                                  Common.VbecsTables.SpecimenTest.LastUpdateDate + " > #" + _currentDateTime.ToString() + "#",
                                                                  Common.VbecsTables.SpecimenTest.TestDate + " DESC");
                DataRow[] rhHistory = dtpatientAboRhHistory.Select("(" + Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = 21 OR " +
                                                                    Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = 68 OR " +
                                                                    Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = 121) AND " +
                                                                    Common.VbecsTables.SpecimenTest.TestResultId + " <> '' AND " +
                                                                    Common.VbecsTables.SpecimenTest.LastUpdateDate + " > #" + _currentDateTime.ToString() + "#",
                                                                    Common.VbecsTables.SpecimenTest.TestDate + " DESC");

             if (aboHistory.Length > 0 && rhHistory.Length > 0)
             {
                 string refreshedABO = aboHistory[0][Common.VbecsTables.BloodUnitTest.TestResultId].ToString().Trim();
                 string refreshedRH = rhHistory[0][Common.VbecsTables.BloodUnitTest.TestResultId].ToString().Trim();

                 if (Common.Utility.GetAboFromString(refreshedABO) != _patient.AboRh.Abo ||
                        Common.Utility.GetRhFromString(refreshedRH) != _patient.AboRh.RH)
                 {
                     return true;
                 }
            }
            return false;
        }


        ///<Developers>
        ///	<Developer>Carl Jensen</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>5/12/2015</CreationDate>
        ///
        /// <summary>
        /// Returns a boolean that indicates if the ABO/Rh changed in the course of working in Issue Unit
        /// CR 3550
        /// </summary>
        public bool DidAntigenNegativeRequirementsChange()
        {
            return _patient.DidAntigenNegativeRequirementsChange();
        }


        ///<Developers>
        ///	<Developer>Carl Jensen</Developer>
        ///</Developers>
        ///<SiteName>Hines OIFO</SiteName>
        ///<CreationDate>5/12/2015</CreationDate>
        ///
        /// <summary>
        /// Returns a boolean that indicates if the Transfusion requirements changed in the course of working in Issue Unit
        /// CR 3550
        /// </summary>
        public bool DidTransfusionRequirementsChange()
        {
            return _patient.DidTransfusionRequirementsChange(_componentRequirementsCache, _allUnits);
        }



		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/11/2006</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8457"> 
		///		<ExpectedInput>Patient with units that can be issued.</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8458"> 
		///		<ExpectedInput>Patient with units that can be issued.</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method was added in response to CR 2081.  It returns a boolean indicating
		/// if there are units available for issue assuming a user with level 3 (lead 
		/// technologist) security credentials.
		/// </summary>
		/// <returns></returns>
		public static int UnitsAvailableForLevelThree(BOL.Patient patient,ArrayList bloodUnitGuids)
		{
			int unitCount = 0;
			BOL.PatientUnitIssue pUI = new BOL.PatientUnitIssue(patient);

			foreach (Guid bloodUnitGuid in bloodUnitGuids)
			{
				if (pUI.UnitSelectable(bloodUnitGuid)) unitCount ++;
			}
			return unitCount;
		}



		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/21/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5994"> 
		///		<ExpectedInput>Guid for a unit that was previously chosen</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5995"> 
		///		<ExpectedInput>Guid for a unit that was not previously chosen</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean value indicating whether or not the unit has been selected.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public bool UnitSelected(Guid bloodUnitGuid)
		{
			try
			{
				_issuedUnits.GetIssuedUnit(bloodUnitGuid);
				return true;
			}
			catch(BOL.BusinessObjectException)
			{
				return false;
			}
		}

		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3806"> 
		///		<ExpectedInput>Create PatientUnitIssue with Patient that has unit available for issue</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3807"> 
		///		<ExpectedInput>Create PatientUnitIssue with Patient that has no units available for issue</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns true if the there are units available for issue (assigned units).
		/// BR_73.20
		/// </summary>
		/// 
		/// <returns></returns>
		public bool UnitsAvailable()
		{
			return _unitsAvailable;
		}

		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3808"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>ArrayList of valid users</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="504"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a list of valid users for the division.
		/// </summary>
		/// <returns></returns>
		public ArrayList GetUserNameList()
		{			
			ArrayList userNames = new ArrayList( _users.Count );

			foreach( BOL.VbecsUser user in _users.Values )
				userNames.Add( user.UserName );

			return userNames;
		}


		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/4/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3922"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>ArrayList of locations</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="494"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a list of Issue To locations based on a VistALink query
		/// </summary>
		/// <returns></returns>
		public ArrayList GetLocations()
		{
			ArrayList locations = new ArrayList();

			// get hospital locations
			DataSet dsLocations = DAL.VAL.HospitalLocation.GetHospitalLocationsByDivision(Common.LogonUser.LogonUserDivisionCode.Trim());
			if (dsLocations.Tables.Count==1)
			{
				_dtLocations = dsLocations.Tables[0];
			}
				
			foreach (DataRow dtRow in _dtLocations.Rows)
			{
				locations.Add(dtRow[Common.DatabaseConstants.VistALinkColumnNames.LocationName].ToString());
			}
				
			return locations;
		}

		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3919"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>ArrayList of clinicians</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="497"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Used to query VistALink to get collection of valid clinicians.
		/// BR_73.01
		/// </summary>
		/// <returns></returns>
		public ArrayList GetOrderingClinicians()
		{
			ArrayList clinicians = new ArrayList();

			string orderingClinician = string.Empty;
			foreach(DataRow dtRow in _dtUnits.Rows)
			{
				string lastName = dtRow.IsNull(Common.VbecsTables.PatientOrder.OrderingProviderLastName) ? string.Empty : dtRow[Common.VbecsTables.PatientOrder.OrderingProviderLastName].ToString();
				string firstName = dtRow.IsNull(Common.VbecsTables.PatientOrder.OrderingProviderFirstName) ? string.Empty : dtRow[Common.VbecsTables.PatientOrder.OrderingProviderFirstName].ToString();
				string middleInitial = dtRow.IsNull(Common.VbecsTables.PatientOrder.OrderingProviderMiddleInitial) ? string.Empty : dtRow[Common.VbecsTables.PatientOrder.OrderingProviderMiddleInitial].ToString();
				
				orderingClinician = lastName + ", " + firstName + " " + middleInitial;
				if (lastName.Trim().Length > 0 && !clinicians.Contains(orderingClinician) )
				{
					clinicians.Add(orderingClinician);
				}
			}
			return clinicians;
		}


		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3925"> 
		///		<ExpectedInput>string with length > 0</ExpectedInput>
		///		<ExpectedOutput>No exception</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="492"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Adds a canned comment that will later be included in the Exception report
		/// </summary>
		/// <param name="comment"></param>
		public void AddIssuedUnitComment(string comment)
		{
			_comments.Add(comment);
		}


		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3926"> 
		///		<ExpectedInput>blood unit with invalid expiration date</ExpectedInput>
		///		<ExpectedOutput>string with warning about invalid expiration date</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="495"> 
		///		<ExpectedInput>invalid bloodunit</ExpectedInput>
		///		<ExpectedOutput>empty string</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets the tooltip messages for the listbox.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public string GetMessagesForTooltip(Guid bloodUnitGuid)
		{
			// Check expiration dates
			CheckExpirations(bloodUnitGuid);

			StringBuilder messageString = new StringBuilder();
			IDictionaryEnumerator enumerator = _unitWarnings.GetEnumerator();

			while (enumerator.MoveNext()) 
			{
				if (bloodUnitGuid != (Guid) enumerator.Key) 
				{
					continue;
				}
				else
				{
					Hashtable messages = (Hashtable) enumerator.Value;
					IDictionaryEnumerator innerEnumerator = messages.GetEnumerator();
					while (innerEnumerator.MoveNext())
					{
						if (((Common.StrRes.StringResourceItem) innerEnumerator.Key).ResString == Common.StrRes.ConfMsg.UC073.RestrictiveUnits().ResString) continue;

						int divider = ((Common.StrRes.StringResourceItem) innerEnumerator.Key).ResString.IndexOf(".");
						string message = ((Common.StrRes.StringResourceItem) innerEnumerator.Key).ResString.Substring(0,divider + 1);
						messageString.Append(message + "\n");
					}
					break;
				}
			}
			return messageString.ToString();
		}


		/// <summary>
		/// Must check this after issue date chosen
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		private void CheckExpirations(Guid bloodUnitGuid)
		{
			BOL.BloodUnit bloodUnit = new BOL.BloodUnit(bloodUnitGuid);
			BOL.Specimen specimen = this.GetSpecimenAssociatedWithUnit(bloodUnit.BloodUnitGuid);
			Hashtable messages = (Hashtable) _unitWarnings[bloodUnit.BloodUnitGuid];

			// BR_73.03 
			// Check specimen expiration date, Level 2
			if (specimen!=null)
			{
				if (specimen.SpecimenExpirationDate < _issueDate)
				{
					if (_bipassSecurityLevelCheck || HasAuthority(Common.FunctionCode.IssueExpiredBloodProducts))
					{
						if (!messages.ContainsKey(Common.StrRes.ConfMsg.UC073.SpecimenExpired())) messages.Add(Common.StrRes.ConfMsg.UC073.SpecimenExpired(), Common.ExceptionType.Unit_issued_on_expired_specimen);
					}
					else
					{
						if (!messages.ContainsKey(Common.StrRes.InfoMsg.UC073.SpecimenExpired())) messages.Add(Common.StrRes.InfoMsg.UC073.SpecimenExpired(), Common.ExceptionType.Unissuable);
					}
				}
			}

			// BR_73.04 
			// Unit expiration date, Level 2
			if (bloodUnit.BloodUnitMedia.UnitExpirationDate < _issueDate)
			{
				if (_bipassSecurityLevelCheck || HasAuthority(Common.FunctionCode.IssueExpiredBloodProducts))
				{
					if (!messages.ContainsKey(Common.StrRes.ConfMsg.UC073.UnitExpired())) messages.Add(Common.StrRes.ConfMsg.UC073.UnitExpired(), Common.ExceptionType.Issued_expired_unit);
				}
				else
				{
					if (!messages.ContainsKey(Common.StrRes.InfoMsg.UC073.UnitExpired())) messages.Add(Common.StrRes.InfoMsg.UC073.UnitExpired(), Common.ExceptionType.Unissuable);
				}
			}
		}


		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/17/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3935"> 
		///		<ExpectedInput>valid blood unit</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="537"> 
		///		<ExpectedInput>invalid bloodunitguid</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean indicating if the unit can be selected for the order.  This
		/// is used for display purposes.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public bool UnitSelectable(Guid bloodUnitGuid)
		{
			StringBuilder messageString = new StringBuilder();
			IDictionaryEnumerator enumerator = _unitWarnings.GetEnumerator();

			while (enumerator.MoveNext()) 
			{
				if (bloodUnitGuid != (Guid) enumerator.Key) 
				{
					continue;
				}
				else
				{
					Hashtable messages = (Hashtable) enumerator.Value;
					IDictionaryEnumerator innerEnumerator = messages.GetEnumerator();
					while (innerEnumerator.MoveNext())
					{
						if (Common.ExceptionType.Unissuable == (Common.ExceptionType) innerEnumerator.Value)
						{
							return false;
						}
					}
					return true;
				}
			}
			return false;
		}


		///<Developers>
		///	<Developer>C Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3936"> 
		///		<ExpectedInput>blood unit</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="493"> 
		///		<ExpectedInput>null for blood unit</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="8485"> 
		///		<ExpectedInput>Selection order invalid</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="8490"> 
		///		<ExpectedInput>Selection order valid</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8486"> 
		///		<ExpectedInput>Expired unit with level 1 access</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="8487"> 
		///		<ExpectedInput>Expired unit with level 2 access</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8488"> 
		///		<ExpectedInput>Expired specimen with level 1 access</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8489"> 
		///		<ExpectedInput>Expired specimen with level 2 access</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Adds a unit for issue and generates error/ warning messages that are sent to the GUI
		/// via an event.
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <param name="emergencyIssue"></param>
		public bool AddUnit(BOL.BloodUnit bloodUnit, bool emergencyIssue)
		{
			if (bloodUnit == null)
			{
				throw new ArgumentException("blood unit");
			}
			// clear out any comments from last unit seleted
			_comments.Clear();
			_unitOK = false;
			Hashtable messages = this.GetMessagesForSelection(bloodUnit.BloodUnitGuid);

			// This message pertains to selection order, so it must be added here (if it's correct or not cannot be known until now):
			if (UnitSelectable(bloodUnit.BloodUnitGuid) && !SelectionOrderValid(bloodUnit))
			{
				if (!messages.Contains(Common.StrRes.ConfMsg.UC073.RestrictiveUnits())) messages.Add(Common.StrRes.ConfMsg.UC073.RestrictiveUnits(), Common.ExceptionType.Less_restrictive_unit_issued);
			}
			else if (messages.Contains(Common.StrRes.ConfMsg.UC073.RestrictiveUnits()))
			{
				messages.Remove(Common.StrRes.ConfMsg.UC073.RestrictiveUnits());
			}

			// BR_73.04 
			// Check unit expiration date, Level 2
			if (bloodUnit.BloodUnitMedia.UnitExpirationDate < _issueDate)
			{
				if (_bipassSecurityLevelCheck || HasAuthority(Common.FunctionCode.IssueExpiredBloodProducts))
				{
					if (!messages.Contains(Common.StrRes.ConfMsg.UC073.UnitExpired())) messages.Add(Common.StrRes.ConfMsg.UC073.UnitExpired(), Common.ExceptionType.Issued_expired_unit);
				}
				else
				{
					messages.Clear(); // Only display the message that explains why the unit cannot be issued
					_allUnitInvalidMessage.Append("\n" + bloodUnit.EyeReadableUnitId + ": " + Common.StrRes.InfoMsg.UC073.UnitExpired().ResString);
					messages.Add(Common.StrRes.InfoMsg.UC073.UnitExpired(), Common.ExceptionType.Unissuable);
				}
			}

			// BR_73.03 
			// Check specimen expiration date, Level 2
			BOL.Specimen specimen = GetSpecimenAssociatedWithUnit(bloodUnit.BloodUnitGuid);
			if (specimen!=null)
			{
				if (specimen.SpecimenExpirationDate < _issueDate)
				{
					if (_bipassSecurityLevelCheck || HasAuthority(Common.FunctionCode.IssueExpiredBloodProducts))
					{
						messages.Add(Common.StrRes.ConfMsg.UC073.SpecimenExpired(), Common.ExceptionType.Unit_issued_on_expired_specimen);
					}
					else
					{
						messages.Clear(); // Only display the message that explains why the unit cannot be issued
						_allUnitInvalidMessage.Append("\n" + bloodUnit.EyeReadableUnitId + ": " + Common.StrRes.InfoMsg.UC073.SpecimenExpired().ResString);
						messages.Add(Common.StrRes.InfoMsg.UC073.SpecimenExpired(), Common.ExceptionType.Unissuable);
					}
				}
			}


			if (messages.Count > 0)	
			{
				IssueMessagesGenerated(messages, bloodUnit);
			}
			else
			{
				_unitOK = true;
			}

			if (!_unitOK)
			{
				return false;
			}
			else 
			{
				DataRow [] dtRowArray = _dtUnits.Select(Common.VbecsTables.BloodUnit.BloodUnitGuid + " = '" + bloodUnit.BloodUnitGuid + "'");  
			
				BOL.IssuedUnit issuedUnit		= new BOL.IssuedUnit();
				issuedUnit.BloodUnitGuid		= bloodUnit.BloodUnitGuid;

				issuedUnit.RequestedBy			= _orderingPhysician;
				issuedUnit.IssueDate			= _issueDate;
				issuedUnit.IssueTechId			= _issuedBy;
				issuedUnit.IssueToLocationName	= _issueToLocation;
				issuedUnit.IssueToLocationIEN	= _issueToLocationIEN;

				issuedUnit.TransporterName		= _transporter;
				issuedUnit.EmergencyIssue		= emergencyIssue;
				issuedUnit.PatientGuid			= _patient.PatientGuid;
				
				issuedUnit.InPatientLocationIndicator = (bool) dtRowArray[0][Common.VbecsTables.PatientTreatment.InPatientIndicator];
				issuedUnit.RemoteStorageIndicator = _remoteStorageIndicator;
				if (_remoteStorageIndicator) issuedUnit.RemoteStorageText = _remoteStorageIdentity;

				issuedUnit.SatisfactoryInspectionIndicator = true;
				issuedUnit.LastUpdateDate = VBECSDateTime.GetDivisionCurrentDateTime();

				_issuedUnits.AddUnit(issuedUnit, _comments);
				RuleBroken("NoUnitsSelected", (_issuedUnits.Count==0));

				return true;
			}
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3937"> 
		///		<ExpectedInput>Previously added unit (satisfactory)</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="7005"> 
		///		<ExpectedInput>Previously added unit (unsatisfactory) and not EI</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="8146"> 
		///		<ExpectedInput>Previously added unit (unsatisfactory) and EI</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="506"> 
		///		<ExpectedInput>null</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///	
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Removes unit from issue.  This is called when a unit is deselected or marked as unsatisfactory.
		/// BR_73.19
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <param name="unsatisfactory"></param>
		public bool RemoveUnit(BOL.BloodUnit bloodUnit, bool unsatisfactory)
		{	
			if (bloodUnit == null)
			{
				throw new ArgumentException("Invalid blood unit");
			}

			bool removed = false;
			DataRow[] drArray = _dtUnits.Select(Common.VbecsTables.BloodUnit.BloodUnitGuid + " = '" + bloodUnit.BloodUnitGuid + "'"); 
			// If the unit is unsatisfactory, remove it.
			if (unsatisfactory)
			{
				if (_units.Contains(bloodUnit))
				{
					_units.Remove(bloodUnit);
					removed = true;
				}
				if (_emergencyUnits.Contains(bloodUnit))
				{
					_emergencyUnits.Remove(bloodUnit);
					removed = true;
				}

				DataTable dtBCEMessages = null;

				//CR 2941
				//CR 2945
				//generate table with BCE_MESSAGE
				//The Unit is marked as unsatisfactory during issue, send BCE message

				Guid assignedPatientGuid = OrderedUnit.GetPatientForOrderedUnit((Guid) drArray[0][Common.VbecsTables.OrderedUnit.OrderedUnitGuid]);
				//CR 3015
				dtBCEMessages = OrderedUnit.UpdateBceForRelease(bloodUnit.BloodUnitGuid, assignedPatientGuid, false);

				//BCE_MESSAGE end
				// quarantines the unit and removes and assignments associated with it
				IssuedUnit.MarkUnitUnsatisfactory( (Guid) drArray[0][Common.VbecsTables.OrderedUnit.OrderedUnitGuid], 
					(Byte []) drArray[0][Common.VbecsTables.OrderedUnit.RowVersion],
					(Byte []) drArray[0][Common.DatabaseConstants.ArtificialColumnNames.UnitStatusRowVersion],
					Common.UpdateFunction.UC073FrmIssueBloodComponents, dtBCEMessages);
				RemoveExceptions(bloodUnit.BloodUnitGuid);
			}
			else
			{
				Hashtable messages = this.GetMessagesForSelection(bloodUnit.BloodUnitGuid);
				if (messages.ContainsKey(Common.StrRes.ConfMsg.UC073.RestrictiveUnits()))
				{
					messages.Remove(Common.StrRes.ConfMsg.UC073.RestrictiveUnits());
				}
				RemoveExceptions(bloodUnit.BloodUnitGuid);
				removed = true;
			}

			_issuedUnits.RemoveUnit(bloodUnit.BloodUnitGuid);
			RuleBroken("NoUnitsSelected", (_issuedUnits.Count==0));
			return removed;
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/15/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3917"> 
		///		<ExpectedInput>Valid workload process ID</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3918"> 
		///		<ExpectedInput>Invalid workload process ID</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Called to save the units that have been chosen for issue.
		/// BR_73.11
		/// </summary>
		public bool Save(Common.WorkloadProcessID workloadProcessId)
		{
			if (workloadProcessId == Common.WorkloadProcessID.Unknown)
			{
				throw new ArgumentException("workloadProcessId");
			}
			CheckExceptions();
				
			return _issuedUnits.SaveUnits(Common.UpdateFunction.UC073FrmIssueBloodComponents, _dtUnits, _dtReleaseUnits, workloadProcessId, _issueExceptions);
		}

		/// <summary>
		/// Checks to make sure that the exceptions generated in the course of unit selection are actually valid.  
		/// See the notes for the RemoveIssueOrderException method for a detailed explanation.
		/// </summary>
		private void CheckExceptions()
		{
			// if there are not autologous or directed units, no need to continue
			if (_autologousUnits == 0 && _directedUnits == 0 && _restrictedUnits == 0) return;

			// I'm calling these up here so they don't have to called multiple times below
			// and cause millions of round trips to the database.
			bool allAutologousUnitsIssued = AllAutologousUnitsIssued();
			bool allDirectedUnitsIssued = AllDirectedUnitsIssued();
			bool allRestrictedUnitsIssues = AllRestrictedUnitsIssued();

			ArrayList issuedUnits = _issuedUnits.GetIssuedUnits();
			foreach (BOL.IssuedUnit issuedUnit in issuedUnits)
			{
				char donationTypeCode = new BOL.BloodUnit(issuedUnit.BloodUnitGuid).DonationType.DonationTypeCode;
				// If unit is autologous, move on!
				if (BOL.BloodUnit.IsAutologous(donationTypeCode))
				{
					continue;
				}
				// If unit is directed, make sure all autologous units selected
				if (BOL.BloodUnit.IsDirected(donationTypeCode))
				{
					if (allAutologousUnitsIssued) RemoveIssueOrderException(issuedUnit.BloodUnitGuid);
				}
					// If unit is restricted, make sure all autologous and directed units selected
				else if (new BOL.BloodUnit(issuedUnit.BloodUnitGuid).DirectedPatient.PatientGuid != Guid.Empty)
				{
					if (allAutologousUnitsIssued && 
						allDirectedUnitsIssued)
					{
						RemoveIssueOrderException(issuedUnit.BloodUnitGuid);
					}
				}
					// If unit is allogeneic, make sure all autologous, directed and restricted units selected
				else
				{
					if (allAutologousUnitsIssued && 
						allDirectedUnitsIssued && 
						allRestrictedUnitsIssues) RemoveIssueOrderException(issuedUnit.BloodUnitGuid);
				}
			}
		}

		/// <summary>
		/// Removes an issue order exception.  This would be called in this example:
		/// There is one autologous and one directed unit available for issue.  The directed unit is chosen
		/// first, so a message is displayed and an exception is generated.  The user then chooses the autologous
		/// unit, so the previous exception is no longer valid.  Call this method to remove! 
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		private void RemoveIssueOrderException(Guid bloodUnitGuid)
		{
			foreach (BOL.ExceptionReport exceptionReport in _issueExceptions)
			{
				if (exceptionReport.ExceptionType == Common.ExceptionType.Less_restrictive_unit_issued &&
					(Guid) exceptionReport.ExceptionData[Common.VbecsTables.ExceptionUnitSelection.UnitGuid] == bloodUnitGuid)
				{
					_issueExceptions.Remove(exceptionReport);
					break;
				}
			}
		}

		/// <summary>
		/// Returns a boolean indicating if all autologous units were issued
		/// </summary>
		/// <returns></returns>
		private bool AllAutologousUnitsIssued()
		{
			int autologousUnitsSelected = 0;
			ArrayList issuedUnits = _issuedUnits.GetIssuedUnits();
			foreach (BOL.IssuedUnit issuedUnit in issuedUnits)
			{
				char donationTypeCode = new BOL.BloodUnit(issuedUnit.BloodUnitGuid).DonationType.DonationTypeCode;
				if (BOL.BloodUnit.IsAutologous(donationTypeCode))
				{
					autologousUnitsSelected++;
				}
			}
			return (autologousUnitsSelected == _autologousUnits);
		}

		/// <summary>
		/// Returns a boolean indicating if all directed units were issued
		/// </summary>
		/// <returns></returns>
		private bool AllDirectedUnitsIssued()
		{
			int directedUnitsSelected = 0;
			ArrayList issuedUnits = _issuedUnits.GetIssuedUnits();
			foreach (BOL.IssuedUnit issuedUnit in issuedUnits)
			{
				char donationTypeCode = new BOL.BloodUnit(issuedUnit.BloodUnitGuid).DonationType.DonationTypeCode;
				if (BOL.BloodUnit.IsDirected(donationTypeCode))
				{
					directedUnitsSelected++;
				}
			}
			return (directedUnitsSelected == _directedUnits);
		}

		/// <summary>
		/// Returns a boolean indicating if all directed units were issued
		/// </summary>
		/// <returns></returns>
		private bool AllRestrictedUnitsIssued()
		{
			int restrictedUnitsSelected = 0;
			ArrayList issuedUnits = _issuedUnits.GetIssuedUnits();
			foreach (BOL.IssuedUnit issuedUnit in issuedUnits)
			{
				char donationTypeCode = new BOL.BloodUnit(issuedUnit.BloodUnitGuid).DonationType.DonationTypeCode;
				if (!BOL.BloodUnit.IsDirected(donationTypeCode) && 
					!BOL.BloodUnit.IsAutologous(donationTypeCode) &&
					new BOL.BloodUnit(issuedUnit.BloodUnitGuid).DirectedPatient.PatientGuid != Guid.Empty)
				{
					restrictedUnitsSelected++;
				}
			}
			return (restrictedUnitsSelected == _restrictedUnits);
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/31/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7210"> 
		///		<ExpectedInput>Valid blood unit guid</ExpectedInput>
		///		<ExpectedOutput>specimen</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7211"> 
		///		<ExpectedInput>Empty Guid</ExpectedInput>
		///		<ExpectedOutput>null</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7212"> 
		///		<ExpectedInput>Invalid guid</ExpectedInput>
		///		<ExpectedOutput>null</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetSpecimenAssociatedWithUnit
		/// </summary>
		/// <returns></returns>
		public BOL.Specimen GetSpecimenAssociatedWithUnit(Guid bloodUnitGuid)
		{
			if (bloodUnitGuid == Guid.Empty)
			{
				throw new ArgumentException("bloodUnitGuid");
			}

			DataRow [] dtRow = _dtUnits.Select(Common.VbecsTables.OrderedUnit.BloodUnitGuid + " = '" + bloodUnitGuid + "'");
			if (dtRow.Length > 0 && dtRow[0][Common.VbecsTables.OrderedComponent.PatientSpecimenGuid] != System.DBNull.Value)
			{
				return new BOL.Specimen( (Guid) dtRow[0][Common.VbecsTables.OrderedComponent.PatientSpecimenGuid]);
			}
			else
			{
				return null;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/31/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7213"> 
		///		<ExpectedInput>Valid blood unit guid</ExpectedInput>
		///		<ExpectedOutput>ordered unit</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7214"> 
		///		<ExpectedInput>Empty guid</ExpectedInput>
		///		<ExpectedOutput>null</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="7215"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>null</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetOrderedUnitAssociatedWithUnit
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public BOL.OrderedUnit GetOrderedUnitAssociatedWithUnit(Guid bloodUnitGuid)
		{
			if (bloodUnitGuid == Guid.Empty)
			{
				throw new ArgumentException("bloodUnitGuid");
			}

			DataRow [] dtRow = _dtUnits.Select(Common.VbecsTables.OrderedUnit.BloodUnitGuid + " = '" + bloodUnitGuid + "'");
			if (dtRow.Length > 0 && dtRow[0][Common.VbecsTables.OrderedUnit.OrderedUnitGuid] != System.DBNull.Value)
			{
				return new BOL.OrderedUnit( (Guid) dtRow[0][Common.VbecsTables.OrderedUnit.OrderedUnitGuid], Common.LogonUser.LogonUserDivisionCode, true);
			}
			else
			{
				return null;
			}
		}


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/31/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7216"> 
		///		<ExpectedInput>Unit that is Emergency Issue</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7217"> 
		///		<ExpectedInput>Unit that is not emergency issue</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean indicating whether or not the unit must be emergency issued.
		/// </summary>
		/// 
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		public bool IsEmergencyIssue(Guid bloodUnitGuid)
		{
			for (int i=0; i<_emergencyUnits.Count; i++)
			{
				if ( ((BOL.BloodUnit) _emergencyUnits[i]).BloodUnitGuid == bloodUnitGuid)
				{
					return true;				
				}
			}
			return false;
		}


		#region private methods

		private SortedList GetUsersSortedList()
		{
			ArrayList usersList = BOL.VbecsUser.GetActiveDivisionUsers( Common.LogonUser.LogonUserDivisionCode );
			SortedList users = new SortedList( usersList.Count );
			
			foreach( BOL.VbecsUser user in usersList )
				if( !users.ContainsKey( user.UserName ) )  
					users.Add( user.UserName, user );

			return users;
		}

		
		/// <summary>
		/// Gets the messages for display when a user selects a unit.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <returns></returns>
		private Hashtable GetMessagesForSelection(Guid bloodUnitGuid)
		{
			StringBuilder messageString = new StringBuilder();
			IDictionaryEnumerator enumerator = _unitWarnings.GetEnumerator();

			while (enumerator.MoveNext()) 
			{
				if (bloodUnitGuid != (Guid) enumerator.Key) 
				{
					continue;
				}
				else
				{
					return  (Hashtable) enumerator.Value;
				}
			}
			return null;
		}

		/// <summary>
		/// Exceptions are generated when the unit is first selected, so they need to be
		/// deleted when the unit is released.
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		private void RemoveExceptions(Guid bloodUnitGuid)
		{
			ArrayList removeExc = new ArrayList();
			for (int i=0; i<_issueExceptions.Count; i++)
			{
				BOL.ExceptionReport excRep = (BOL.ExceptionReport) _issueExceptions[i];
				if ( (Guid) excRep.ExceptionData[Common.VbecsTables.ExceptionUnitIssue.UnitGuid] == bloodUnitGuid)
				{
					removeExc.Add(excRep);
				}	
			}

			for (int j=0; j<removeExc.Count; j++)
			{
				BOL.ExceptionReport excRep = (BOL.ExceptionReport) removeExc[j];
				_issueExceptions.Remove(excRep);
			}
		}


		/// <summary>
		/// This sorts the units into 2 groups.  The first group consists of
		/// units that can be issued regularly.  The second group consists of
		/// units that must be emergency issued.
		/// BR_73.02
		/// 
		/// This method is a critical area, but is tested by the SelectionOrderValid 
		/// unit tests.  It is also called by almost every other unit test.
		/// </summary>
		private void SortUnits()
		{
			foreach(DataRow dtRow in _dtUnits.Rows)
			{
				BOL.BloodUnit bloodUnit = new BOL.BloodUnit( (Guid) dtRow[Common.VbecsTables.OrderedUnit.BloodUnitGuid]);
				// Lock blood unit status here
				BOL.OrderedUnit orderedUnit = new BOL.OrderedUnit(dtRow);
				// Lock ordered unit here
				bool specimenRequiredIndicator = (bool) dtRow[Common.VbecsTables.OrderedComponent.SpecimenRequiredIndicator];

				BOL.Specimen specimen = null;
				if (dtRow[Common.VbecsTables.OrderedComponent.PatientSpecimenGuid] != System.DBNull.Value)
				{
					// CR2899: start
					if(SpecimensCache.ContainsKey( (Guid) dtRow[Common.VbecsTables.OrderedComponent.PatientSpecimenGuid]))
					{
						specimen = (BOL.Specimen) SpecimensCache[ (Guid) dtRow[Common.VbecsTables.OrderedComponent.PatientSpecimenGuid] ];
					}
					else
					{
						specimen = new BOL.Specimen( (Guid) dtRow[Common.VbecsTables.OrderedComponent.PatientSpecimenGuid]);
						SpecimensCache.Add(specimen.SpecimenGuid, specimen);
					}
					// CR2899: end
				}
				if (BOL.BloodUnit.IsAutologous(bloodUnit.DonationType.DonationTypeCode))
				{
					_autologousUnits++;
				}
				else if (BOL.BloodUnit.IsDirected(bloodUnit.DonationType.DonationTypeCode))
				{
					_directedUnits++;
				}
				else if (bloodUnit.DirectedPatient.PatientGuid != Guid.Empty)
				{
					_restrictedUnits++;
				}
    		
				this.SetMessages(bloodUnit, orderedUnit, specimenRequiredIndicator, specimen);
			}
			if (_emergencyUnits.Count > 1) _emergencyUnits = OrderUnits(_emergencyUnits);
			if (_units.Count > 1) _units = OrderUnits(_units);
		}


		/// <summary>
		/// This sorts the units so they appear in correct order on the picklist.
		/// Expiration date is already sorted by the sproc, so just sort component 
		/// classes here.
		/// Implements BR_73.07, 
		/// </summary>
		/// 
		/// <param name="units"></param>
		private ArrayList OrderUnits(ArrayList units)
		{
			int[] compClassArray = {(int) Common.ComponentClass.RBC, 
									   (int) Common.ComponentClass.WB, 
									   (int) Common.ComponentClass.FFP, 
									   (int) Common.ComponentClass.PLT,
									   (int) Common.ComponentClass.CRYO,
									   (int) Common.ComponentClass.Other};

			ArrayList sortedUnits = new ArrayList(units.Count);
			foreach (int componentClass in compClassArray)
			{
				Order(componentClass, units, sortedUnits);
				if (sortedUnits.Count == units.Count) break;
			}
			return sortedUnits;
		}


		private void Order(int componentClassID, ArrayList units, ArrayList sortedUnits)
		{
			foreach (BOL.BloodUnit bloodUnit in units)
			{
				if (bloodUnit.BloodComponentClassId == componentClassID) sortedUnits.Add(bloodUnit);
			}
		}


		// Just shortening the message a little for the tooltip
		private string GetShortMessage(Common.StrRes.StringResourceItem infoMsg)
		{
			int period = infoMsg.ResString.IndexOf(".")+1;
			if (period > 1)
			{
				return infoMsg.ResString.Substring(0,period);
			}
			else
			{
				return infoMsg.ResString;
			}
		}


		/// <summary>
		/// Retrieves and stores all of the messages.  They are stored because the messages
		/// are displayed multiple times in the applications.  Keeping them in memory will 
		/// prevent trips to the DB.
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <param name="orderedUnit"></param>
		/// <param name="specimenRequiredIndicator"></param>
		/// <param name="specimen"></param>
		private void SetMessages(BOL.BloodUnit bloodUnit, BOL.OrderedUnit orderedUnit, bool specimenRequiredIndicator, BOL.Specimen specimen)
		{
			Hashtable messages = ValidateUnit(bloodUnit, orderedUnit, specimenRequiredIndicator, specimen);
			_unitWarnings.Add(bloodUnit.BloodUnitGuid, messages);
		}
		
		///<summary>
		/// Verifies that the user has authority to issue unit.
		///</summary>
		private bool HasAuthority(Common.FunctionCode function)
		{
			return _user.IsAuthorizedForFunction(function);
		}

		#endregion

		

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/7/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7234"> 
		///		<ExpectedInput>Valid arguments</ExpectedInput>
		///		<ExpectedOutput>No warning messages</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="8175"> 
		///		<ExpectedInput>Valid arguments, but null specimen</ExpectedInput>
		///		<ExpectedOutput>One warning (EI)</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="7243"> 
		///		<ExpectedInput>Valid arguments, EI</ExpectedInput>
		///		<ExpectedOutput>No warning messages</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="7244"> 
		///		<ExpectedInput>Valid arguments, EI</ExpectedInput>
		///		<ExpectedOutput>No warning messages</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8180"> 
		///		<ExpectedInput>Valid, FFP</ExpectedInput>
		///		<ExpectedOutput>1 error</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7235"> 
		///		<ExpectedInput>Invalid unit - part one</ExpectedInput>
		///		<ExpectedOutput>Warning messages for BRs 73.23, 3.05, 73.23, 73.35</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="7236"> 
		///		<ExpectedInput>Invalid unit - part two</ExpectedInput>
		///		<ExpectedOutput>Warning messages for BRs 73.15, 73.05, 73.26</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="7237"> 
		///		<ExpectedInput>Invalid unit - part three</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7240"> 
		///		<ExpectedInput>Invalid unit - part four - insufficient priveleges</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="7241"> 
		///		<ExpectedInput>Valid - part four - sufficient priveleges</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="7242"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This validates the unit and generates warning/ error messages.  If it is determined that
		/// the user cannot select the unit, all other messages are cleared (why display informational 
		/// messages if the unit cannot be selected?).  Only the message that explains why the unit 
		/// cannot be chosen will be displayed eliminating confusion.  
		/// Not selectable: BR_73.27, BR_73.15, BR_73.05
		/// Selectable with (may require override): BR_73.14, BR_73.28, BR_73.03, BR_73.04, BR_73.16
		/// Informational only: BR_73.12, BR_73.30, BR_73.31, BR_73.32, BR_73.33, BR_73.35
		/// BR_73.06, BR_73.23, BR_73.26
		/// 
		/// CR2789: Code that checked for ConversionDiscrepancy was removed from first "if" statement:
		///			_patient.ABORhSource == Common.ABORhSource.ConversionDiscrepancy
		///			
		/// </summary>
		public Hashtable ValidateUnit(BOL.BloodUnit bloodUnit, BOL.OrderedUnit orderedUnit, bool specimenRequiredIndicator, BOL.Specimen specimen)
		{
			Hashtable messages = new Hashtable();
			bool mustEmergencyIssue = false;
			bool noCrossmatchResults = false;
     

			// CR2899: start
			// indicates if the current specimen has been TAS tested (performance enhancement)
			bool specimenTASTested = false;
			// indicates if the current specimen has had inconsistent ABO/Rh results, repeat or historic (performance enhancement)
			bool specimenABORhTestResultsInconsistent = false;
		
			if (PatientABORh == Common.ABORhSource.None) _patientABORh = _patient.ABORhSource;
		
			// Cache all specimen information here
			if (specimen != null)
			{
                // specimen hasn't been cached yet
                if (!TASSpecimenResultsCache.ContainsKey(specimen.SpecimenGuid))
                {
                    specimenTASTested = specimen.TASTested;
                    TASSpecimenResultsCache.Add(specimen.SpecimenGuid, specimenTASTested);
                    specimenABORhTestResultsInconsistent = (PatientABORh == Common.ABORhSource.SpecimenTestDiscrepancy ||
                                                        PatientABORh == Common.ABORhSource.ConversionDiscrepancy ||
                                                        specimen.RepeatABORhTestsMatch == Common.TypingTestResults.Discrepancy);
                    SpecimenABORhTestResultsInconsistentCache.Add(specimen.SpecimenGuid, specimenABORhTestResultsInconsistent);
                } 
				// specimen has been cached
				else
				{
					specimenTASTested = (bool) TASSpecimenResultsCache[specimen.SpecimenGuid];
					specimenABORhTestResultsInconsistent = (bool) SpecimenABORhTestResultsInconsistentCache[specimen.SpecimenGuid];
				}
			}
			//CR2899: end

		
			Common.ComponentClass componentClass = (Common.ComponentClass) bloodUnit.BloodComponentClassId;

			#region unitsNotSelectable // There are errors that result in the unit being unselectable under any circumstance

			
			#region Units may be selectable if they qualify for Emergency issue

			// CR2899: start
			bool canEI = bloodUnit.EligibleForEmergencyIssue(
				orderedUnit.CrossmatchRequiredIndicator, 
				specimenABORhTestResultsInconsistent,
				specimen != null && specimen.ABORhTested);
			// CR2899: end


			
			// BR_73.23 Check to see if order requires specimen testing:
			if (specimenRequiredIndicator)
			{
				// Next, is there a specimen associated with the order and if so,
				// is testing complete:
				if (specimen == null || !specimenTASTested)
				{
					if (canEI) 
					{
						mustEmergencyIssue = true; // message is generated below
					}
					else 
					{
						_allUnitInvalidMessage.Append("\n" + bloodUnit.EyeReadableUnitId + ": " + GetShortMessage(Common.StrRes.InfoMsg.UC073.SpecimenTestingNotCompleted()));
						messages.Add(Common.StrRes.InfoMsg.UC073.SpecimenTestingNotCompleted(), Common.ExceptionType.Unissuable);
					}
				}
			}



			// BR_3.05, TT_52.03: Patient must have specimen test results and ABO/Rh may not 
			// be inconclusive
			if (PatientABORh == Common.ABORhSource.Conversion
				|| PatientABORh == Common.ABORhSource.None
				|| _patient.AboRh.Abo == Common.ABO.I 
				|| _patient.AboRh.RH == Common.RH.Inconclusive)
			{
				if (canEI) 
				{
					mustEmergencyIssue = true;
				}
				else 
				{		
					_allUnitInvalidMessage.Append("\n" + bloodUnit.EyeReadableUnitId + ": " + GetShortMessage(Common.StrRes.InfoMsg.UC073.NoABORhResultsOnSpecimen()));
					messages.Add(Common.StrRes.InfoMsg.UC073.NoABORhResultsOnSpecimen(), Common.ExceptionType.Unissuable);
				}
			}

			// BR_73.23 Check crossmatch results - This checks to make sure the unit wasn't crossmatched as compatible.
			if (orderedUnit.CrossmatchRequiredIndicator)
			{
				Common.CrossmatchResult result = Common.Utility.GetCrossmatchResultEnumFromString(orderedUnit.CurrentCrossmatchResultCode); 

				if (result != Common.CrossmatchResult.CompatibleEXM &&
					result != Common.CrossmatchResult.CompatibleSafeToTransfuse &&
					result != Common.CrossmatchResult.IncompatibleDirectorApprovalNeeded)
				{
					if (canEI) 
					{
						mustEmergencyIssue = true; // message is generated below
						noCrossmatchResults = true;
					}
					else 
					{
						if (result == Common.CrossmatchResult.Unknown)
						{
							_allUnitInvalidMessage.Append("\n" + bloodUnit.EyeReadableUnitId + ": " + GetShortMessage(Common.StrRes.InfoMsg.UC073.NoCrossmatchResults()));
							messages.Add(Common.StrRes.InfoMsg.UC073.NoCrossmatchResults(), Common.ExceptionType.Unissuable);
						}
					}
				}

				
				// BR_73.35
				if (result == Common.CrossmatchResult.Inconclusive)
				{
					_allUnitInvalidMessage.Append("\n" + bloodUnit.EyeReadableUnitId + ": " + GetShortMessage(Common.StrRes.InfoMsg.UC073.InconclusiveCrossmatch()));
					messages.Add(Common.StrRes.InfoMsg.UC073.InconclusiveCrossmatch(), Common.ExceptionType.Unissuable);
				}
			}


			//CR2789, CR2899 start
			//BR_73.05: Compare ABO/Rh of specimen test against other specimen tests and patient record
			if (specimen != null && specimenABORhTestResultsInconsistent)
			{
				if (canEI)
				{
					mustEmergencyIssue = true;
				}
				else
				{
					// BR_73.15
					// Compare the ABO/Rh of repeat tests on the specimen
					if (specimen.RepeatABORhTestsMatch == Common.TypingTestResults.Discrepancy )
					{
						_allUnitInvalidMessage.Append("\n" + bloodUnit.EyeReadableUnitId + ": " + GetShortMessage(Common.StrRes.InfoMsg.UC073.ABORhRepeatDiscrepancy()));
						messages.Add(Common.StrRes.InfoMsg.UC073.ABORhRepeatDiscrepancy(), Common.ExceptionType.Unissuable);
					}
					// BR_73.05
					// Compare ABO/Rh of specimen test against other specimen tests and patient record
					else
					{
						_allUnitInvalidMessage.Append("\n" + bloodUnit.EyeReadableUnitId + ": " + GetShortMessage(Common.StrRes.InfoMsg.UC073.ABORhDiscrepancy()));
						messages.Add(Common.StrRes.InfoMsg.UC073.ABORhDiscrepancy(), Common.ExceptionType.Unissuable);
					}
				}
			}
			// CR2789, CR2899 end


			// BR_73.37, TT_73.01, TT_73.02, TT_73.03 - This has to be called for both Full service and Transfusion only facilities since the patient's
			// ABO/Rh may have changed since selection
			bool valid = true; 

			if (componentClass == Common.ComponentClass.WB) valid = this.ValidateWholeBloodTT_7301(bloodUnit, false);
			else if	(componentClass == Common.ComponentClass.RBC) valid = this.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, false);
			else if (componentClass == Common.ComponentClass.FFP) valid = this.ValidatePlasmaTT73_03(bloodUnit, false);
		{
			if (!valid)
			{
				_allUnitInvalidMessage.Append("\n" + bloodUnit.EyeReadableUnitId + ": " + Common.StrRes.ConfMsg.UC073.AboIncompatibleUnitIssued().ResString);
				messages.Add(Common.StrRes.ConfMsg.UC073.AboIncompatibleUnitIssued(), Common.ExceptionType.Unissuable);
			}
		}


			if (mustEmergencyIssue) 
			{
				_emergencyUnits.Add(bloodUnit);	
			}
			else
			{
				_units.Add(bloodUnit);
			}
			// No need to continue if unit cannot be issued
			if (messages.Count > 0) return messages;


			#endregion

            // BR_73.26 (Unit ABO/Rh confirmaion; method deleted) - Note: This BR can never really be triggered since a unit cannot be assigned 
            // unless confirmations are complete.  Furthermore, testing results may not be invalidated while unit is assigned.

			

			

			
			// BR_73.06
			if (bloodUnit.BloodUnitStatus.QuarantineIndicator)
			{
				_allUnitInvalidMessage.Append("\n" + bloodUnit.EyeReadableUnitId + ": " + GetShortMessage(Common.StrRes.InfoMsg.UC073.UnitQuarantined()));
				messages.Add(Common.StrRes.InfoMsg.UC073.UnitQuarantined(), Common.ExceptionType.Unissuable);
				return messages;
			}


			// Check for bad crossmatch results - When a unit is crossmatched to either of these statuses, it
			// should be released from patient assignment (BR_40.14).  So, this code should never be called!
			if (orderedUnit.CrossmatchRequiredIndicator)
			{
				Common.CrossmatchResult result = Common.Utility.GetCrossmatchResultEnumFromString(orderedUnit.CurrentCrossmatchResultCode); 
				if (result == Common.CrossmatchResult.CompatibleDontTransfuse ||
					result == Common.CrossmatchResult.IncompatibleUnsafeToTransfuse)
				{
					_allUnitInvalidMessage.Append("\n" + bloodUnit.EyeReadableUnitId + ": " + GetShortMessage(Common.StrRes.InfoMsg.UC073.UnitCrossmatchBad()));
					messages.Add(Common.StrRes.InfoMsg.UC073.UnitCrossmatchBad(), Common.ExceptionType.Unissuable);
					return messages;
				}	
			}


			// BR_73.27
			// Frozen units cannot be issued!
			if (bloodUnit.IsFrozen)
			{
				_allUnitInvalidMessage.Append("\n" + bloodUnit.EyeReadableUnitId + ": " + GetShortMessage(Common.StrRes.InfoMsg.UC073.UnitFrozen()));
				messages.Add(Common.StrRes.InfoMsg.UC073.UnitFrozen(), Common.ExceptionType.Unissuable);
				return messages;
			}

			#endregion  



			#region overrideRequired // There are errors that require the user to have a higher security level to override

			if (_isFullService)
			{
				// BR_73.33
				if (componentClass == Common.ComponentClass.WB) valid = this.ValidateWholeBloodTT_7301(bloodUnit, true);
				else if	(componentClass == Common.ComponentClass.RBC) valid = this.ValidateNonWholeBloodNonPlasmaTT_7302(bloodUnit, true);
				else if (componentClass == Common.ComponentClass.FFP) valid = this.ValidatePlasmaTT73_03(bloodUnit, true);
				{
					if (!valid)
					{
						messages.Add(Common.StrRes.ConfMsg.UC073.AboIncompatibleUnitIssuedSystemInterp(), Common.ExceptionType.ABO_discrepant_unit_issued);
					}
				}
			}

            // CR3260
            // The next couple methods require an arraylist as input, so I'm creating it here. Note that the better 
            // solution would be to just pass in the unit datatable that we already have, but I don't want to touch
            // critical code for this bug fix.
            if (_allUnits == null)
            {
                _allUnits = new ArrayList();
                for (int i = 0; i < _dtUnits.Rows.Count; i++)
                {
                    BOL.BloodUnit bu = new BOL.BloodUnit((Guid)_dtUnits.Rows[i]["BloodUnitGuid"]);
                    _allUnits.Add(bu);
                }
            }
            // CR3260 end



            // CR2899 start
            // The AntNegCache is a hashtable used to store antigen negative requierments. It will always be empty
            // the first time through. Even if there are no AN requirements, the hashtable count will reflect the 
            // number of untis the second time through, therefore preventting redundant calls.
            //
            // Test 1: Make sure it functions correctly with multiple units.
            if (AntNegCache.Count == 0)
            {
                _antNegCache = _patient.ValidateAntibodyAntigenRequirementsMet(_allUnits);
            }
            ArrayList antigenNegReqs = (ArrayList)AntNegCache[bloodUnit.BloodUnitGuid];
            // CR2899 end

			// BR_73.14, Level 3
			// Clinically significant Antibody or Antigen negative requirements
			if (componentClass == Common.ComponentClass.WB ||
				componentClass == Common.ComponentClass.RBC )
			{
				foreach (BOL.AntigenNegativeMessage aNM in antigenNegReqs)
				{
					if (!aNM.Tested  || (aNM.Tested && aNM.TestResult == Common.TestResult.P) )
					{			
						if (_bipassSecurityLevelCheck || HasAuthority(Common.FunctionCode.IssueAntigenPosOrUntypedRedbloodcells))
						{
							if (aNM.AntibodyName != string.Empty)
							{
								messages.Add(Common.StrRes.ConfMsg.UC073.AntibodyProblem(aNM.AntibodyName, bloodUnit.EyeReadableUnitId, aNM.AntigenName), Common.ExceptionType.Antigen_positive_untested_units_issued);
							}
							else
							{
								messages.Add(Common.StrRes.ConfMsg.UC073.AntigenProblem(aNM.AntigenName, bloodUnit.EyeReadableUnitId), Common.ExceptionType.Antigen_positive_untested_units_issued);
							}
						}
						else
						{
							messages.Clear();  // Only display the message that explains why the unit cannot be issued
							if (aNM.AntibodyName != string.Empty)
							{
								_allUnitInvalidMessage.Append("\n" + Common.StrRes.InfoMsg.UC073.AntibodyProblem(aNM.AntibodyName, bloodUnit.EyeReadableUnitId, aNM.AntigenName).ResString);
								messages.Add(Common.StrRes.InfoMsg.UC073.AntibodyProblem(aNM.AntibodyName, bloodUnit.EyeReadableUnitId, aNM.AntigenName), Common.ExceptionType.Unissuable);
							}
							else
							{
								_allUnitInvalidMessage.Append("\n" + Common.StrRes.InfoMsg.UC073.AntigenProblem(aNM.AntigenName, bloodUnit.EyeReadableUnitId).ResString);
								messages.Add(Common.StrRes.InfoMsg.UC073.AntigenProblem(aNM.AntigenName, bloodUnit.EyeReadableUnitId), Common.ExceptionType.Unissuable);
							}
							return messages;
						}
					}
				}
			}

			
			// BR_73.28
			// Check Transfusion Requirements, Level 3
			// TRs: TT_3.04

            // CR2899 start 
            if (_componentRequirementsCache == null)
            {
                _componentRequirementsCache = _patient.ValidateComponentRequirements(_allUnits);
            }
            ArrayList cRs = (ArrayList)ComponentRequirementsCache[bloodUnit.BloodUnitGuid];
            // CR2899 end

			if (cRs != null && cRs.Count > 0)
			{
				StringBuilder cRString = new StringBuilder();
				foreach (string cR in cRs)
				{
					if (cRString.Length > 0) cRString.Append(", ");
					cRString.Append(cR);
				}

				if (_bipassSecurityLevelCheck || HasAuthority(Common.FunctionCode.IssueAntigenPosOrUntypedRedbloodcells))
				{
					messages.Add(Common.StrRes.ConfMsg.UC073.TRNotSatisfied(cRString.ToString()), Common.ExceptionType.Unit_issued_with_unsatisfied_transfusion_requirement);
				}
				else
				{
					messages.Clear(); // Only display the message that explains why the unit cannot be issued
					_allUnitInvalidMessage.Append("\n" + bloodUnit.EyeReadableUnitId + ": " + Common.StrRes.InfoMsg.UC073.TRNotSatisfied(cRString.ToString()).ResString);
					messages.Add(Common.StrRes.InfoMsg.UC073.TRNotSatisfied(cRString.ToString()), Common.ExceptionType.Unissuable);
					return messages;
				}
			}

			
			// BR_73.16, Level 1
			// Rh negative patient/ positive unit
            // CR3261
            if (!_patient.IsUnitRhSatisfactory(bloodUnit)) 

			{
				messages.Add(Common.StrRes.ValidMsg.UC003.BR321(), Common.ExceptionType.Antigen_positive_untested_units_issued);
			}

			#endregion
			

			#region informationalOnly

			// BR_73.12 
			// Check to see if unit is assigned to someone else
			ArrayList rowVersions = new ArrayList();
			DataTable dtOrderedUnits = BOL.OrderedUnit.GetOrderedUnit(bloodUnit.BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode);
			if (dtOrderedUnits.Rows.Count > 1)
			{
				messages.Add(Common.StrRes.InfoMsg.UC073.UnitIsAssigned(), Common.ExceptionType.Unknown);

				// We must keep track of all of these because the units that are assigned to 
				// other patients will have to be released if the unit is issued.
				foreach (DataRow dtRowOU in dtOrderedUnits.Rows)
				{
					AddReleaseEntry(dtRowOU);
				}
			}

			
			// BR_73.30
			// Biohazardous
			if (bloodUnit.BloodUnitStatus.BiohazardousWarningIndicator)
			{
				messages.Add(Common.StrRes.ConfMsg.UC073.Biohazardous(), Common.ExceptionType.Biohazardous_unit_issued);
			}

			
            // BR_73.31
            // CR2899 start
            if (_emergencyUnits.Contains(bloodUnit) && (specimenRequiredIndicator && (specimen == null || !specimenTASTested || noCrossmatchResults))
                ||
                orderedUnit.CrossmatchRequiredIndicator && noCrossmatchResults) // For other orders when a crossmatch is required
            // CR2899 end
            {
                messages.Add(Common.StrRes.ConfMsg.UC073.EmergencyIssue(), Common.ExceptionType.Emergency_issue_required_testing_incomplete);
            }

			// CR2889 start
			// BR_73.05, BR_73.15: specimen discrepancy 
			if (canEI && specimenABORhTestResultsInconsistent)
			{
				messages.Add(Common.StrRes.InfoMsg.UC073.EmergencyIssueSpecimenDiscrepancy(), Common.ExceptionType.Unknown);
			}
			// CR2889 end

						
			#endregion

			// If we've made it down to here, the unit is issuable 
			_unitsAvailable = true;
			
			return messages;
		}


		#region TTs

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7222"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7223"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements the logic described in TT_73.01 regarding the selection
		/// of blood units that are whole blood.
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <param name="systemInterp"></param>
		/// <returns></returns>
		public bool ValidateWholeBloodTT_7301(BOL.BloodUnit bloodUnit, bool systemInterp)
		{
			// System ABO is the system's interpretation, not the user's.
			Common.ABO abo = Common.ABO.NA;
			if (systemInterp)
			{
				abo = Common.Utility.GetAboFromString(this.GetSystemAbo());
			}
			else 
			{
				abo = _patient.AboRh.Abo;
			}

			Common.ABO bloodUnitABO = bloodUnit.BloodUnitMedia.AboRh.Abo;
			bool validUnit = false;

			switch(abo)
			{
				case Common.ABO.O:
				{
					if(bloodUnitABO == Common.ABO.O)
					{
						validUnit = true;
					}
					break;
				}
				case Common.ABO.A:
				{
					if(bloodUnitABO == Common.ABO.A)
					{
						validUnit = true;
					}
					break;
				}
				case Common.ABO.B:
				{
					if(bloodUnitABO == Common.ABO.B)
					{
						validUnit = true;
					}
					break;
				}
				case Common.ABO.AB:
				{
					if(bloodUnitABO == Common.ABO.AB)
					{
						validUnit = true;
					}
					break;
				}
				default:
				{
					validUnit = false;
					break;
				}
			}
			return validUnit;
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7224"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7225"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements the logic described in TT_73.02 regarding the selection
		/// of blood units that are not whole blood or plasma.
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <param name="systemInterp"></param>
		/// <returns>bool indicating whether this unit is valid for selection</returns>
		public bool ValidateNonWholeBloodNonPlasmaTT_7302(BOL.BloodUnit bloodUnit, bool systemInterp)
		{
			Common.ABO abo = Common.ABO.NA;
			if (systemInterp)
			{
				abo = Common.Utility.GetAboFromString(this.GetSystemAbo());
			}
			else 
			{
				abo = _patient.AboRh.Abo;
			}

			Common.ABO bloodUnitABO = bloodUnit.BloodUnitMedia.AboRh.Abo;
			bool validUnit = false;

			switch(abo)
			{
				case Common.ABO.O:
				{
					if(bloodUnitABO == Common.ABO.O)
					{
						validUnit = true;
					}
					break;
				}
				case Common.ABO.A:
				{
					if(bloodUnitABO == Common.ABO.A || bloodUnitABO == Common.ABO.O)
					{
						validUnit = true;
					}
					break;
				}
				case Common.ABO.B:
				{
					if(bloodUnitABO == Common.ABO.B || bloodUnitABO == Common.ABO.O)
					{
						validUnit = true;
					}
					break;
				}
				case Common.ABO.AB:
				{
					if(bloodUnitABO == Common.ABO.AB || bloodUnitABO == Common.ABO.O || 
						bloodUnitABO == Common.ABO.A || bloodUnitABO == Common.ABO.B)
					{
						validUnit = true;
					}
					break;
				}
				default:
				{
					if (bloodUnitABO == Common.ABO.O)
					{
						validUnit = true;
					}
					break;
				}
			}
			return validUnit;
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/1/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7226"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7227"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Implements the logic described in TT_73.03 regarding the selection
		/// of plasma units.
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <param name="systemInterp"></param>
		/// <returns>bool indicating whether this unit is valid for selection</returns>
		public bool ValidatePlasmaTT73_03(BOL.BloodUnit bloodUnit, bool systemInterp)
		{
			Common.ABO abo = Common.ABO.NA;
			if (systemInterp)
			{
				abo = Common.Utility.GetAboFromString(this.GetSystemAbo());
			}
			else 
			{
				abo = _patient.AboRh.Abo;
			}

			Common.ABO bloodUnitABO = bloodUnit.BloodUnitMedia.AboRh.Abo;
			bool validUnit = false;

			switch(abo)
			{
				case Common.ABO.O:
				{
					validUnit = true;
					break;
				}
				case Common.ABO.A:
				{
					if(bloodUnitABO == Common.ABO.A || bloodUnitABO == Common.ABO.AB)
					{
						validUnit = true;
					}
					break;
				}
				case Common.ABO.B:
				{
					if(bloodUnitABO == Common.ABO.B || bloodUnitABO == Common.ABO.AB)
					{
						validUnit = true;
					}
					break;
				}
				case Common.ABO.AB:
				{
					if(bloodUnitABO == Common.ABO.AB)
					{
						validUnit = true;
					}
					break;
				}
				default:
				{
					if (bloodUnitABO == Common.ABO.AB)
					{
						validUnit = true;
					}
					break;
				}
			}
			return validUnit;
		}

		#endregion
	
		/*
		 *
		 
		start 2789

		/// <summary>
		/// Checks to see that Patient ABO/Rh results are consistent.  Compares the current specimen to 
		/// the historic, stopping the comparison if justified results are found.
		/// BR_73.05
		/// </summary>
		/// <param name="specimen"></param>
		/// <returns></returns>
		private bool SpecimenTestsResultsInconsistent(BOL.Specimen specimen)
		{
			return specimen.ABORhTestsMatchHistoricResults() == Common.TypingTestResults.Discrepancy;
		}
		
		end 2789
		
		*/

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7579"> 
		///		<ExpectedInput>Choose the autologous unit (most restrictive)</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="8153"> 
		///		<ExpectedInput>Choose the directed unit after the autologous 
		///		unit has been chosen</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8154"> 
		///		<ExpectedInput>Choose the restricted unit after the autologous and 
		///		directed units have been chosen.</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="8434"> 
		///		<ExpectedInput>Choose the allogeneic unit after the autologous, 
		///		directed and restricted units have been chosen.</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7580"> 
		///		<ExpectedInput>Restriced when there is an autologous unit available</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8151"> 
		///		<ExpectedInput>Allogeneic when there is an autologous unit available</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8152"> 
		///		<ExpectedInput>Allogeneic when there is a directed unit available</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="0" testid ="8435"> 
		///		<ExpectedInput>Allogeneic when there is a directed unit available</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8436"> 
		///		<ExpectedInput>Restricted unit when autologous available</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8437"> 
		///		<ExpectedInput>Restricted unit when directed available</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean indicating whether unit is being selected in the correct order.  More 
		/// restrictive units must be chosen first.
		/// BR_73.25
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <returns></returns>
		public bool SelectionOrderValid(BOL.BloodUnit bloodUnit)
		{
			Common.DonationType donationType = Common.Utility.GetDonationTypeCodeEnumFromChar(bloodUnit.DonationType.DonationTypeCode);
			// BR_73.25, Level 1

			//  Unit is autologous, everything is fine
			if (BOL.BloodUnit.IsAutologous(bloodUnit.DonationType.DonationTypeCode))
			{
				return true;
			}
				// Unit is directed, make sure no autologous units are out there
			else if (BOL.BloodUnit.IsDirected(bloodUnit.DonationType.DonationTypeCode))
			{
				// 1) Get all units of the same component class
				// 2) Make sure they have a more restrictive donation type
				// 3) Make sure it hasn't already been selected.
				DataRow [] dtRowArray = _dtUnits.Select(Common.VbecsTables.ComponentClass.ComponentClassId + " = " + bloodUnit.BloodComponentClassId); 
				foreach (DataRow dtRow in dtRowArray)
				{
					char donationTypeCode = Convert.ToChar(dtRow[Common.VbecsTables.DonationType.DonationTypeCode].ToString());
					if (BOL.BloodUnit.IsAutologous(donationTypeCode))
					{
						try
						{
							if (_issuedUnits.GetIssuedUnit((Guid) dtRow[Common.VbecsTables.BloodUnit.BloodUnitGuid]) != null)
							{
								continue;
							}
						}
						catch(BOL.BusinessObjectException)
						{
							return false;
						}
					}
				}
			}

				// Unit is restricted, make sure no autologous or directed units are out there
			else if (bloodUnit.DirectedPatient.PatientGuid != Guid.Empty)
			{
				// 1) Get all units of the same component class
				// 2) Make sure they have a more restrictive donation type
				// 3) Make sure it hasn't already been selected.
				DataRow [] dtRowArray = _dtUnits.Select(Common.VbecsTables.ComponentClass.ComponentClassId + " = " + bloodUnit.BloodComponentClassId); 
				foreach (DataRow dtRow in dtRowArray)
				{
					char donationTypeCode = Convert.ToChar(dtRow[Common.VbecsTables.DonationType.DonationTypeCode].ToString());
					if (BOL.BloodUnit.IsDirected(donationTypeCode) || BOL.BloodUnit.IsAutologous(donationTypeCode))
					{
						try
						{
							if (_issuedUnits.GetIssuedUnit((Guid) dtRow[Common.VbecsTables.BloodUnit.BloodUnitGuid]) != null)
							{
								continue;
							}
						}
						catch(BOL.BusinessObjectException)
						{
							return false;
						}
					}
				}
			}
				// Unit is allogeneic. make sure no autologous, directed or restricted units are out there
			else
			{
				// 1) Get all units of the same component class
				// 2) Make sure they have a more restrictive donation type
				// 3) Make sure it hasn't already been selected.
				DataRow [] dtRowArray = _dtUnits.Select(Common.VbecsTables.ComponentClass.ComponentClassId + " = " + bloodUnit.BloodComponentClassId); 
				foreach (DataRow dtRow in dtRowArray)
				{
					char donationTypeCode = Convert.ToChar(dtRow[Common.VbecsTables.DonationType.DonationTypeCode].ToString());
					//					if (BOL.BloodUnit.IsAutologous( Convert.ToChar(dtRow[Common.VbecsTables.DonationType.DonationTypeCode].ToString())) || 
					//						IsRestricted( Common.Utility.GetDonationTypeCodeEnumFromChar( Convert.ToChar(dtRow[Common.VbecsTables.DonationType.DonationTypeCode].ToString()))))
					if ( !dtRow.IsNull(Common.VbecsTables.BloodUnitMedia.DirectedPatientGuid) && (Guid) dtRow[Common.VbecsTables.BloodUnitMedia.DirectedPatientGuid] != Guid.Empty || 
						BOL.BloodUnit.IsDirected(donationTypeCode) || 
						BOL.BloodUnit.IsAutologous(donationTypeCode))
					{
						try
						{
							if (_issuedUnits.GetIssuedUnit( (Guid) dtRow[Common.VbecsTables.BloodUnit.BloodUnitGuid]) != null)
							{
								continue;
							}
						}
						catch(BOL.BusinessObjectException)
						{
							return false;
						}
					}
				}
			}
			return true;
		}


		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6701"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6702"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// InitializeReleaseDataTable
		/// </summary>
		/// <returns></returns>
		public static DataTable InitializeReleaseDataTable()
		{
			DataTable dtOrderedUnitsForRelease  = new DataTable();
			dtOrderedUnitsForRelease.Columns.Add(Common.VbecsTables.OrderedUnit.BloodUnitGuid, typeof(Guid));
			dtOrderedUnitsForRelease.Columns.Add(Common.VbecsTables.OrderedUnit.ReleaseTechId, typeof(string));
			dtOrderedUnitsForRelease.Columns.Add(Common.VbecsTables.OrderedUnit.ReleaseDate, typeof(DateTime));
			dtOrderedUnitsForRelease.Columns.Add(Common.VbecsTables.OrderedUnit.LastUpdateUser, typeof(string));
			dtOrderedUnitsForRelease.Columns.Add(Common.VbecsTables.OrderedUnit.LastUpdateFunctionId, typeof(int));
			dtOrderedUnitsForRelease.Columns.Add(Common.VbecsTables.OrderedUnit.RowVersion, typeof(byte[]));
			dtOrderedUnitsForRelease.Columns.Add(Common.VbecsTables.OrderedUnit.DivisionCode, typeof(string));
			return dtOrderedUnitsForRelease;
		}

		private void AddReleaseEntry(DataRow dtRow)
		{
			Guid orderedUnitGUID = (Guid) dtRow[Common.VbecsTables.OrderedUnit.OrderedUnitGuid];
			if ( _dtUnits.Select(Common.VbecsTables.OrderedUnit.OrderedUnitGuid + " = '" + orderedUnitGUID + "'").Length == 1 ) return;

			if (_dtReleaseUnits == null) _dtReleaseUnits = BOL.PatientUnitIssue.InitializeReleaseDataTable();
			DataRow dtRowRelease = _dtReleaseUnits.NewRow();
			dtRowRelease[Common.VbecsTables.OrderedUnit.BloodUnitGuid]	= (Guid) dtRow[Common.VbecsTables.OrderedUnit.BloodUnitGuid];
			dtRowRelease[Common.VbecsTables.OrderedUnit.ReleaseTechId]	= _issuedBy;
			dtRowRelease[Common.VbecsTables.OrderedUnit.ReleaseDate]	= _issueDate;
			dtRowRelease[Common.VbecsTables.OrderedUnit.LastUpdateUser] = Common.LogonUser.LogonUserName;
			dtRowRelease[Common.VbecsTables.OrderedUnit.LastUpdateFunctionId] = (int) Common.UpdateFunction.UC073FrmIssueBloodComponents;
			dtRowRelease[Common.VbecsTables.OrderedUnit.RowVersion]		= (byte[]) dtRow[Common.VbecsTables.OrderedUnit.RowVersion];
			_dtReleaseUnits.Rows.Add(dtRowRelease);
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7581"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7582"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// LoadDataRowFromThis
		/// </summary>
		/// <param name="dtRow"></param>
		/// <returns></returns>
		public override DataRow LoadDataRowFromThis(DataRow dtRow)
		{
			throw new BOL.BusinessObjectException("LoadDataRowFromThis is not implemented!");
		}

		/// <summary>
		/// LoadFromDataRow
		/// </summary>
		/// <param name="dtRow"></param>
		protected override void LoadFromDataRow(DataRow dtRow)
		{
			throw new BOL.BusinessObjectException("LoadFromDataRow is not implemented!");
		}



		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/16/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6705"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6706"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Beep Beep
		/// </summary>
		/// <param name="message"></param>
		/// <returns></returns>
		public static bool Beep(Common.StrRes.StringResourceItem message)
		{
			if (message.ResString.Trim() == Common.StrRes.ConfMsg.UC073.SpecimenExpired().ResString.Trim() ||
				message.ResString.Trim() == Common.StrRes.ConfMsg.UC073.UnitExpired().ResString.Trim() || 
				message.ResString.Substring(0,20) == Common.StrRes.ConfMsg.UC073.AntibodyProblem(string.Empty, string.Empty, string.Empty).ResString.Substring(0,20) ||
				message.ResString.Substring(0,20) == Common.StrRes.ConfMsg.UC073.AntigenProblem(string.Empty, string.Empty).ResString.Substring(0,20) ||
				message.ResString.Trim() == Common.StrRes.ValidMsg.UC003.BR321().ResString.Trim() ||
				message.ResString.Trim() == Common.StrRes.ConfMsg.UC073.RestrictiveUnits().ResString.Trim() ||
				message.ResString.Trim() == Common.StrRes.InfoMsg.UC073.UnitFrozen().ResString.Trim() ||
				message.ResString.Substring(0,20) == Common.StrRes.ConfMsg.UC073.TRNotSatisfied(string.Empty).ResString.Substring(0,20) ||
                //CR3260: This is a bad fix, but no easy way to resolve at the moment. This method needs to be refactored so it doesn't depend on message string content.
                message.ResString.IndexOf("Transfusion Requirement(s) is not satisfied.") > -1 ||
				message.ResString.Trim() == Common.StrRes.ConfMsg.UC073.Biohazardous().ResString.Trim() ||
				message.ResString.Trim() == Common.StrRes.ConfMsg.UC073.EmergencyIssue().ResString.Trim() ||
				message.ResString.Trim() == Common.StrRes.ConfMsg.UC073.AboIncompatibleUnitIssued().ResString.Trim())
			{
				return true;
			}
			else
			{
				return false;
			}
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/7/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7230"> 
		///		<ExpectedInput>bloodUnitGuid with valid ABO results</ExpectedInput>
		///		<ExpectedOutput>ABO other than I</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7231"> 
		///		<ExpectedInput>bloodUnitGuid with valid ABO results</ExpectedInput>
		///		<ExpectedOutput>ABO of I</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns the system ABO which may be different from the user's
		/// </summary>
		/// <returns></returns>
        public string GetSystemAbo()
		{
		    // CR2899 start
		    if (SystemABORhCache.ContainsKey(_patient.PatientGuid))
		    {
		        return (string) SystemABORhCache[_patient.PatientGuid];
		    }
		    // CR 2899 end

		    // Get the specimen responsible for the patient's current ABO/Rh
		    DataRow dtRow = DAL.Patient.GetPatientABORh(_patient.PatientGuid);
		    if (dtRow.IsNull(Common.VbecsTables.PatientSpecimen.PatientSpecimenGuid)) return string.Empty;


		    BOL.Specimen specimen = new BOL.Specimen((Guid) dtRow[Common.VbecsTables.PatientSpecimen.PatientSpecimenGuid]);

		    string antiA = string.Empty;
		    string antiB = string.Empty;
		    string antiAB = string.Empty;
		    string a1Cells = string.Empty;
		    string bCells = string.Empty;

		    try
		    {
		        antiA =
		            specimen.TestResults.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " +
		                                        (int) Common.TestType.AntiA_TAS + " OR " +
		                                        Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " +
		                                        (int) Common.TestType.AntiA)[0][Common.VbecsTables.SpecimenTest.TestResultId].
		                ToString().Trim();
		        antiB =
		            specimen.TestResults.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " +
		                                        (int) Common.TestType.AntiB_TAS + " OR " +
		                                        Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " +
		                                        (int) Common.TestType.AntiB)[0][Common.VbecsTables.SpecimenTest.TestResultId].
		                ToString().Trim();
		        if (
		            specimen.TestResults.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " +
		                                        (int) Common.TestType.AntiAB_TAS + " OR " +
		                                        Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " +
		                                        (int) Common.TestType.AntiAB).Length > 0)
		        {
		            antiAB =
		                specimen.TestResults.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " +
		                                            (int) Common.TestType.AntiAB_TAS + " OR " +
		                                            Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " +
		                                            (int) Common.TestType.AntiAB)[0][
		                                                Common.VbecsTables.SpecimenTest.TestResultId].ToString().Trim();
		        }
		        else
		        {
		            antiAB = "X";
		        }
		        a1Cells =
		            specimen.TestResults.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " +
		                                        (int) Common.TestType.A1Cells + " OR " +
		                                        Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " +
		                                        (int) Common.TestType.PlasmaA1Cells)[0][
		                                            Common.VbecsTables.SpecimenTest.TestResultId].ToString().Trim();
		        bCells =
		            specimen.TestResults.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " +
		                                        (int) Common.TestType.BCells + " OR " +
		                                        Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " +
		                                        (int) Common.TestType.PlasmaBCells)[0][
		                                            Common.VbecsTables.SpecimenTest.TestResultId].ToString().Trim();
		    }
		    catch (Exception)
		    {
		        SystemABORhCache.Add(_patient.PatientGuid, string.Empty);
		        return string.Empty;
		    }

		    //CR 3289
            if (antiA == string.Empty || antiB == string.Empty || antiAB == string.Empty || a1Cells == string.Empty ||
                bCells == string.Empty)
            {

                SystemABORhCache.Add(_patient.PatientGuid, string.Empty);
                return string.Empty;
            }

            else
            {
                SystemABORhCache.Add(_patient.PatientGuid, Common.Utility.InterpretAbo(antiA, antiB, antiAB, a1Cells, bCells));
                return Common.Utility.InterpretAbo(antiA, antiB, antiAB, a1Cells, bCells);
            }
		}


	    ///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/7/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7232"> 
		///		<ExpectedInput>bloodUnitGuid with valid Rh results</ExpectedInput>
		///		<ExpectedOutput>Rh other than I</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="7233"> 
		///		<ExpectedInput>bloodUnitGuid with invalid Rh results</ExpectedInput>
		///		<ExpectedOutput>Rh of I</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetSystemRh
		/// </summary>
		/// <returns></returns>
		public string GetSystemRh()
		{
            //CR 3289
            string antiD = string.Empty;
            string dControl = string.Empty;
         
            DataRow dtRow = DAL.Patient.GetPatientABORh(_patient.PatientGuid);
            if (dtRow.IsNull(Common.VbecsTables.PatientSpecimen.PatientSpecimenGuid))
            {
                return string.Empty;
            }
            else
            {
                BOL.Specimen specimen = new BOL.Specimen((Guid)dtRow[Common.VbecsTables.PatientSpecimen.PatientSpecimenGuid]);

                //CR 3435
                try
                {
                    antiD = specimen.TestResults.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + (int)Common.TestType.AntiD_TAS + " OR " + Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + (int)Common.TestType.AntiD)[0][Common.VbecsTables.SpecimenTest.TestResultId].ToString().Trim();
                    if (specimen.TestResults.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + (int)Common.TestType.DControl + " OR " + Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + (int)Common.TestType.ControlD).Length > 0)
                    {
                        dControl = specimen.TestResults.Select(Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + (int)Common.TestType.DControl + " OR " + Common.VbecsTables.SpecimenTest.BloodTestTypeId + " = " + (int)Common.TestType.ControlD)[0][Common.VbecsTables.SpecimenTest.TestResultId].ToString().Trim();
                    }
                    else
                    {
                        //if DControl was disabled setting it to X
                        dControl = "X";
                    }
                }
                catch (Exception)
                {
                    //if AntiD or DControl was left blank
                    return string.Empty;
                }

                if (antiD == string.Empty || dControl == string.Empty)
                    return string.Empty;
                else
                    return Common.Utility.InterpretRh(antiD, dControl);
            }
		}


		#endregion


		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/17/2003</CreationDate>
		///
		/// <summary>
		/// This inner class manages the units selected for issue.
		/// </summary>
		public class IssuedUnits :ICollection
		{

			#region variables
			
			/// <summary>
			/// Holds the units that have been chosen for issue.
			/// </summary>
			private ArrayList _issuedUnits;
			/// <summary>
			/// Holds all of the comments entered for a unit.
			/// </summary>
			private DataTable _dtComments;
			/// <summary>
			/// Date of issue
			/// </summary>
			private DateTime _issueDate;
			/// <summary>
			/// Collection to hold ExceptionReport entries
			/// </summary>
			private ArrayList exceptions;

			#endregion


			#region constructors

			///<Developers>
			///	<Developer>Rob Heiberger</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>2/16/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6707"> 
			///		<ExpectedInput>Valid issue date</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6708"> 
			///		<ExpectedInput>Invalid issue date</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			///<summary>
			/// This is the main constructor.
			///</summary>
			public IssuedUnits(DateTime issueDate)
			{
				if (issueDate == DateTime.MinValue || issueDate == DateTime.MaxValue)
				{
					throw new ArgumentException("issueDate");
				}

				_issueDate = issueDate;
				_issuedUnits = new ArrayList();
				_dtComments = GetIssuedUnitCommentTable();
				exceptions = new ArrayList();
			}

			#endregion


			#region methods

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/13/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7583"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7584"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// GetIssuedUnits
			/// </summary>
			/// <returns></returns>
			public ArrayList GetIssuedUnits()
			{
				return _issuedUnits;
			}

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/13/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7585"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7586"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// Adds a unit to collection.
			/// </summary>
			/// <param name="issuedUnit"></param>
			/// <param name="comments"></param>
			public void AddUnit(BOL.IssuedUnit issuedUnit, ArrayList comments)
			{
				_issuedUnits.Add(issuedUnit);
				AddComments(issuedUnit, comments);
			}

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/13/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7587"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7588"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// Removes a unit from IssuedUnits.  
			/// </summary>
			/// <param name="bloodUnitGuid"></param>
			public void RemoveUnit(Guid bloodUnitGuid)
			{
				foreach (BOL.IssuedUnit issuedUnit in _issuedUnits)
				{
					if (issuedUnit.BloodUnitGuid == bloodUnitGuid)
					{
						_issuedUnits.Remove(issuedUnit);
						RemoveComments(issuedUnit.IssuedUnitGuid);
						break;
					}
				}
			}
			
			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/13/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7589"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7590"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			///<summary>
			/// Retrieves an issued unit from the collection.
			///</summary>
			public BOL.IssuedUnit GetIssuedUnit(Guid bloodUnitGuid)
			{
				for (int i=0; i<_issuedUnits.Count; i++)
				{
					if (bloodUnitGuid == ((BOL.IssuedUnit) _issuedUnits[i]).BloodUnitGuid)
					{
						return (BOL.IssuedUnit) _issuedUnits[i];
					}
				}
				throw new BOL.BusinessObjectException("Unit not found");
			}

			///<Developers>
			///	<Developer>C. Jensen</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>3/8/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6911"> 
			///		<ExpectedInput>Valid arguments</ExpectedInput>
			///		<ExpectedOutput>true</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="6912"> 
			///		<ExpectedInput>Invalid arguments</ExpectedInput>
			///		<ExpectedOutput>false</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// Saves the units for one Issued Unit.  Note that one patient may have > 1 units.
			/// BR_73.29
			/// </summary>
			public bool SaveUnits(Common.UpdateFunction lastUpdateFunctionId, 
				DataTable dtUnits, 
				DataTable dtAllReleaseUnits, 
				Common.WorkloadProcessID workloadProcessId,
				ArrayList issueExceptions)
			{
				DataTable dtWorkloadEvents = null;

				if (lastUpdateFunctionId == Common.UpdateFunction.Invalid ||			
					workloadProcessId == Common.WorkloadProcessID.Unknown ||
					issueExceptions == null)				
				{
					throw new ArgumentException();
				}

				Guid [] patientGuids = new Guid[1];
				ArrayList addedSpecimens = new ArrayList();
				
				// Create DataTables for insertion of units

				DataTable dtIssueUnits = new DataTable();
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.BloodUnitGuid, typeof(Guid));
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.InPatientLocationIndicator, typeof(bool));
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.IssueDateTime, typeof(DateTime));
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.IssuedUnitGuid, typeof(Guid));
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.IssueSatisfactoryIndicator, typeof(bool));
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.IssueTechId, typeof(string));
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.IssueToLocationName, typeof(string));
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.LastUpdateDate, typeof(DateTime));
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.LastUpdateFunctionId, typeof(int));
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.LastUpdateUser, typeof(string));
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.RemoteStorageIndicator, typeof(bool));
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.RemoteStorageText, typeof(string));
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.PhysicianName, typeof(string));
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.TransporterName, typeof(string));
				dtIssueUnits.Columns.Add(Common.VbecsTables.IssuedUnit.PatientGuid, typeof(Guid));
				dtIssueUnits.Columns.Add(Common.VbecsTables.VamcDivision.DivisionCode, typeof(string));

				DataTable dtStatusUnits = DAL.BloodUnitStatus.GetEmptyBloodUnitStatusTableSchema(true);
				DataTable dtReleaseUnits = BOL.PatientUnitIssue.InitializeReleaseDataTable();
                			
				DataTable dtSpecimens = BOL.Specimen.GetSpecimen(string.Empty);
				dtSpecimens.Rows.Clear();
				
				
				foreach (BOL.IssuedUnit issuedUnit in _issuedUnits)
				{		
					patientGuids[0] = issuedUnit.PatientGuid;
					BOL.Patient patient = new Patient(issuedUnit.PatientGuid);
					
					// CR2899 start
					DataRow [] dtUnitRow = dtUnits.Select(Common.VbecsTables.OrderedUnit.BloodUnitGuid + " = '" + issuedUnit.BloodUnitGuid + "'");
					BOL.OrderedUnit orderedUnit = new BOL.OrderedUnit( (Guid) dtUnitRow[0][Common.VbecsTables.OrderedUnit.OrderedUnitGuid], Common.LogonUser.LogonUserDivisionCode, true);
					// CR2899 end
										
					
					//Create the workload datatable
					//CR2222
					if(dtWorkloadEvents == null)
					{			
						if(orderedUnit.OrderedTestGuid == System.Guid.Empty)
						{
							dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, Guid.Empty, orderedUnit.OrderedComponentGuid, patientGuids, Common.WorkloadTransactionType.Patient, false);
						}
						else
						{
							dtWorkloadEvents = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, orderedUnit.OrderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Patient, false);
						}
					}
					else
					{
						DataTable dtWorkloadEventsTemp = null;
						if(orderedUnit.OrderedTestGuid == System.Guid.Empty)
						{
							dtWorkloadEventsTemp = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, Guid.Empty, orderedUnit.OrderedComponentGuid, patientGuids, Common.WorkloadTransactionType.Patient, false);
						}
						else
						{
							dtWorkloadEventsTemp = BOL.WorkloadEvent.GenerateWorkloadData(workloadProcessId, orderedUnit.OrderedTestGuid, Guid.Empty, patientGuids, Common.WorkloadTransactionType.Patient, false);
						}
						
						foreach(DataRow dtRowWE in dtWorkloadEventsTemp.Rows)
						{	
							DataRow newRow = dtWorkloadEvents.NewRow();
							newRow.ItemArray = dtRowWE.ItemArray;
							dtWorkloadEvents.Rows.Add(newRow);
						}
					}
					//END - CR2222
				
					DataRow dtRow = dtIssueUnits.NewRow();
					dtRow[Common.VbecsTables.IssuedUnit.LastUpdateFunctionId] = (int) lastUpdateFunctionId;
					dtRow[Common.VbecsTables.VamcDivision.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					dtIssueUnits.Rows.Add(issuedUnit.LoadDataRowFromThis(dtRow));

					DataRow [] dtRowArray = dtUnits.Select(Common.VbecsTables.BloodUnit.BloodUnitGuid + " = '" + dtRow[Common.VbecsTables.BloodUnit.BloodUnitGuid].ToString() + "'");

					if (issuedUnit.EmergencyIssue)
					{
						System.Guid bloodUnitGuid = (System.Guid)dtRowArray[0][Common.VbecsTables.BloodUnitStatus.BloodUnitGuid];

						BOL.BloodUnitStatus tmpBUS = new BOL.BloodUnitStatus(bloodUnitGuid);
						DataRow dtStatusRow = tmpBUS.LoadDataRowFromThis(dtStatusUnits.NewRow());

						dtStatusRow[Common.VbecsTables.BloodUnitStatus.EmergencyIssueIndicator] = issuedUnit.EmergencyIssue;
						dtStatusUnits.Rows.Add(dtStatusRow);
					}

					if (dtAllReleaseUnits != null)
					{
						AddReleaseRows(dtAllReleaseUnits, dtReleaseUnits, 
							(Guid) dtRowArray[0][Common.VbecsTables.BloodUnitStatus.BloodUnitGuid],
							issuedUnit.IssueTechId,
							issuedUnit.IssueDate);
					}

					if (!dtRowArray[0].IsNull(Common.VbecsTables.PatientSpecimen.PatientSpecimenGuid))
					{
						DateTime specimenExpiration = (DateTime) dtRowArray[0][Common.VbecsTables.PatientSpecimen.SpecimenExpirationDate];
						// BR_73.29
						if (specimenExpiration > _issueDate.AddDays(3) && !addedSpecimens.Contains( (Guid) dtRowArray[0][Common.VbecsTables.PatientSpecimen.PatientSpecimenGuid]))
						{
							BOL.Specimen specimen = new Specimen( (Guid) dtRowArray[0][Common.VbecsTables.PatientSpecimen.PatientSpecimenGuid] );
							DataRow dtSpecimenRow = dtSpecimens.NewRow();
							specimen.LoadDataRowFromThis(dtSpecimenRow);

							dtSpecimenRow[Common.VbecsTables.PatientSpecimen.SpecimenExpirationDate] = _issueDate.AddDays(3);
							if (!dtSpecimens.Columns.Contains(Common.VbecsTables.PatientSpecimen.LastUpdateFunctionId))
							{
								dtSpecimens.Columns.Add(Common.VbecsTables.PatientSpecimen.LastUpdateFunctionId, typeof(int));
							}
							dtSpecimenRow[Common.VbecsTables.PatientSpecimen.LastUpdateFunctionId] = (int) lastUpdateFunctionId;
							if (!dtSpecimens.Columns.Contains(Common.VbecsTables.PatientSpecimen.LastUpdateUser))
							{
								dtSpecimens.Columns.Add(Common.VbecsTables.PatientSpecimen.LastUpdateUser, typeof(string));
							}
							dtSpecimenRow[Common.VbecsTables.PatientSpecimen.LastUpdateUser] = Common.LogonUser.LogonUserName;
							dtSpecimens.Rows.Add(dtSpecimenRow);
							addedSpecimens.Add(specimen.SpecimenGuid);
						}
					}
				}

				
				dtWorkloadEvents = Common.Utility.AppendLastUpdateInformation(dtWorkloadEvents, lastUpdateFunctionId);

				DataTable dtIssueExceptions = null;

				foreach(BOL.ExceptionReport exceptionReport in issueExceptions)
				{
					if (dtIssueExceptions == null)
						dtIssueExceptions = exceptionReport.ExceptionData.Table.Clone();
					dtIssueExceptions.Rows.Add(exceptionReport.ExceptionData.ItemArray);
				}


				dtStatusUnits = Common.Utility.AppendLastUpdateInformation(dtStatusUnits, lastUpdateFunctionId);
				
				//the list of patient and unit guids for which we will send Release message to BCE
				ArrayList releaseAssignmentsForBCE = new ArrayList();

				foreach (BOL.IssuedUnit issuedUnit in _issuedUnits)
				{		
					//getting all patients assigned to a unit
					DataTable dtPatients = DAL.PatientAssignments.GetPatientsAssignmentDetails(issuedUnit.BloodUnitGuid);

					for (int j=0; j< dtPatients.Rows.Count; j++)
					{
						if (issuedUnit.PatientGuid.ToString() != dtPatients.Rows[j][Common.VbecsTables.Patient.PatientGuid].ToString())
						{
							releaseAssignmentsForBCE.Add(issuedUnit.BloodUnitGuid);
							releaseAssignmentsForBCE.Add((Guid)dtPatients.Rows[j][Common.VbecsTables.Patient.PatientGuid]);
						}
					}
				}


				DataTable dtBCEMessages = null;
				DataTable dtTempBCEMessages = null;

				//CR 2941
				//CR 2945
				//generate table with BCE_MESSAGE
				//The Unit is issued, send BCE message

				foreach (BOL.IssuedUnit issuedUnit in _issuedUnits)
				{	
					DataRow [] dtUnitRow = dtUnits.Select(Common.VbecsTables.OrderedUnit.BloodUnitGuid + " = '" + issuedUnit.BloodUnitGuid + "'");
					BOL.OrderedUnit orderedUnit = new BOL.OrderedUnit( (Guid) dtUnitRow[0][Common.VbecsTables.OrderedUnit.OrderedUnitGuid], Common.LogonUser.LogonUserDivisionCode, true);
					//CR 3015
					if (dtBCEMessages == null)
					{
						dtBCEMessages = OrderedUnit.UpdateBce(issuedUnit.BloodUnitGuid, issuedUnit.PatientGuid, orderedUnit.OrderedComponentGuid, issuedUnit.IssueToLocationName, Common.HL7BPDispenseStatus.DS, false);
					}
					else
					{
						dtTempBCEMessages = OrderedUnit.UpdateBce(issuedUnit.BloodUnitGuid, issuedUnit.PatientGuid, orderedUnit.OrderedComponentGuid, issuedUnit.IssueToLocationName, Common.HL7BPDispenseStatus.DS, false);
						if (dtTempBCEMessages.Rows.Count > 0)
							dtBCEMessages.Rows.Add(dtTempBCEMessages.Rows[0].ItemArray);
					}
				}

				//send message to release unit for other patients

				if (releaseAssignmentsForBCE.Count > 0)
				{
					for (int i=0; i<releaseAssignmentsForBCE.Count; i=i+2)
					{
						//CR 3015
						if (dtBCEMessages == null)
						{
							dtBCEMessages = OrderedUnit.UpdateBceForRelease((Guid)releaseAssignmentsForBCE[i], (Guid)releaseAssignmentsForBCE[i+1], false);
						}
						else
						{
							dtTempBCEMessages = OrderedUnit.UpdateBceForRelease((Guid)releaseAssignmentsForBCE[i], (Guid)releaseAssignmentsForBCE[i+1], false);
							if (dtTempBCEMessages.Rows.Count > 0)
								dtBCEMessages.Rows.Add(dtTempBCEMessages.Rows[0].ItemArray);
						}
						
					}
				}

				//BCE_MESSAGE end
				
				// Call the update				
				bool result = BOL.IssuedUnit.Save(dtIssueUnits, dtReleaseUnits, dtStatusUnits, dtSpecimens, _dtComments, dtWorkloadEvents, dtIssueExceptions, dtBCEMessages);


				return result;
			}

			private void AddReleaseRows(DataTable dtAllReleaseUnits, DataTable dtReleaseUnits, Guid bloodUnitGUID, string issuedBy, DateTime issueDate)
			{
				DataRow [] drRowArray = dtAllReleaseUnits.Select(Common.VbecsTables.OrderedUnit.BloodUnitGuid + " = '" + bloodUnitGUID + "'");
				foreach (DataRow dtRow in drRowArray)
				{
					dtRow[Common.VbecsTables.OrderedUnit.ReleaseTechId] = issuedBy;
					dtRow[Common.VbecsTables.OrderedUnit.ReleaseDate] = issueDate;
					dtRow[Common.VbecsTables.OrderedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					dtReleaseUnits.Rows.Add(dtRow.ItemArray);
				}
			}


			/// <summary>
			/// Adds comments related to a particular issued unit.
			/// </summary>
			/// <param name="issuedUnit"></param>
			/// <param name="comments"></param>
			private void AddComments(BOL.IssuedUnit issuedUnit, ArrayList comments)
			{
				foreach (string comment in comments)
				{
					DataRow dtRow = _dtComments.NewRow();

					dtRow[ISSUEUNITCOMMENT.IssuedUnitCommentGuid] = System.Guid.NewGuid();
					dtRow[ISSUEUNITCOMMENT.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
					dtRow[ISSUEUNITCOMMENT.LastUpdateFunctionId] = (int) Common.UpdateFunction.UC073FrmIssueBloodComponents;
					dtRow[ISSUEUNITCOMMENT.IssuedUnitCommentText] = comment;
					dtRow[ISSUEUNITCOMMENT.IssuedUnitGuid] = issuedUnit.IssuedUnitGuid;
					dtRow[ISSUEUNITCOMMENT.LastUpdateUser] = Common.LogonUser.LogonUserName;
					_dtComments.Rows.Add(dtRow);
				}
			}


			private DataTable GetIssuedUnitCommentTable()
			{
				_dtComments = new DataTable(ISSUEUNITCOMMENT.TableName);
				_dtComments.Columns.Add(ISSUEUNITCOMMENT.IssuedUnitCommentGuid, typeof(System.Guid));
				_dtComments.Columns.Add(ISSUEUNITCOMMENT.DivisionCode, typeof(string));
				_dtComments.Columns.Add(ISSUEUNITCOMMENT.LastUpdateFunctionId, typeof(int));
				_dtComments.Columns.Add(ISSUEUNITCOMMENT.IssuedUnitCommentText, typeof(string));
				_dtComments.Columns.Add(ISSUEUNITCOMMENT.IssuedUnitGuid, typeof(Guid));
				_dtComments.Columns.Add(ISSUEUNITCOMMENT.LastUpdateUser, typeof(string));
				return _dtComments;
			}


			private void RemoveComments(Guid issuedUnitGUID)
			{
				DataRow [] drArray = _dtComments.Select(ISSUEUNITCOMMENT.IssuedUnitGuid + " = '" + issuedUnitGUID + "'");
				foreach (DataRow dr in drArray)
				{
					_dtComments.Rows.Remove(dr);
				}
			}

			#endregion


			#region Implementation of ICollection

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/13/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7591"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7592"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// This copies the collection to an array (not implemented).
			/// </summary>
			/// <param name="array"></param>
			/// <param name="index"></param>
			public void CopyTo(System.Array array, int index)
			{
				throw new BOL.BusinessObjectException("This method has not been implemented!");
			}

			///<Developers>
			///	<Developer>Carrie Van Stedum</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/26/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6518"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="6519"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// This indicates whether or not the collection is synchronized.  This is not implemented
			/// and is not needed since a user can only access one ordered component at a time.
			/// </summary>
			public bool IsSynchronized
			{
				get
				{
					return false;
				}
			}
            
		
			///<Developers>
			///	<Developer>Carrie Van Stedum</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>1/26/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="6520"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///<Case type="1" testid ="6521"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			///<summary>
			/// This returns the number of blood units associated with an order.
			///</summary>
			public int Count
			{
				get
				{
					return _issuedUnits.Count;
				}
			}

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/13/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7593"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7594"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// This has not been implemented.
			/// </summary>
			public object SyncRoot
			{
				get
				{
					throw new BOL.BusinessObjectException("This method has not been implemented!");
				}
			}

			#endregion
            

			#region Implementation of IEnumerable

			///<Developers>
			///	<Developer>Greg Lohse</Developer>
			///</Developers>
			///<SiteName>Hines OIFO</SiteName>
			///<CreationDate>8/13/2005</CreationDate>
			///<TestCases>
			///	
			///<Case type="0" testid ="7595"> 
			///		<ExpectedInput>Valid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///<Case type="1" testid ="7596"> 
			///		<ExpectedInput>Invalid Parameters</ExpectedInput>
			///		<ExpectedOutput>NA</ExpectedOutput>
			///	</Case>
			///
			///</TestCases>
			///<Update></Update>
			///<ArchivePlan></ArchivePlan>
			///<Interfaces></Interfaces>
			///
			/// <summary>
			/// Returns the enumerator for the collection.
			/// </summary>
			/// <returns></returns>
			public System.Collections.IEnumerator GetEnumerator()
			{
				return new IssuedUnitEnumerator(this);
			}

			#endregion


			#region Enumerator class
			
			///<summary>
			///This class in an implementation of the enumerator needed for the SelectedUnits 
			///class.
			///</summary>
			public class IssuedUnitEnumerator : IEnumerator
			{
				private IssuedUnits _issuedUnits;		// A copy of the collection
				private int _index;						// track the index of the current element
				private Object currentElement;			// the current element, returned by Current

				internal IssuedUnitEnumerator(IssuedUnits issuedUnits) 
				{
					_issuedUnits = issuedUnits;
					_index = 0;
					currentElement = _issuedUnits._issuedUnits;

					if (_issuedUnits._issuedUnits.Count == 0) _index = -1;
				}

				///<Developers>
				///	<Developer>Greg Lohse</Developer>
				///</Developers>
				///<SiteName>Hines OIFO</SiteName>
				///<CreationDate>8/13/2005</CreationDate>
				///<TestCases>
				///	
				///<Case type="0" testid ="7597"> 
				///		<ExpectedInput>Valid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///<Case type="1" testid ="7598"> 
				///		<ExpectedInput>Invalid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///</TestCases>
				///<Update></Update>
				///<ArchivePlan></ArchivePlan>
				///<Interfaces></Interfaces>
				///
				///<summary>
				/// Implementation of MoveNext
				/// </summary>
				public virtual bool MoveNext() 
				{
					if (_index < 0) 
					{  // check to see if you can take this action. False indicates
						// that the MoveNext did not work
						currentElement = _issuedUnits._issuedUnits;
						return false;
					}
					
					int placeHolder = 0;
					foreach (BOL.IssuedUnit issuedUnit in this._issuedUnits)
					{
						if (placeHolder == this._index)
						{
							currentElement = issuedUnit;
							_index++;
							placeHolder++;

							//once at the end of our collection, ensure the user cannot MoveNext
							if (_index == _issuedUnits._issuedUnits.Count) 
							{
								_index = -1;
							}
							return true;
						}
						else 
						{
							placeHolder++;
						}
					}
					return false;
				}

				///<Developers>
				///	<Developer>Greg Lohse</Developer>
				///</Developers>
				///<SiteName>Hines OIFO</SiteName>
				///<CreationDate>8/13/2005</CreationDate>
				///<TestCases>
				///	
				///<Case type="0" testid ="7599"> 
				///		<ExpectedInput>Valid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///<Case type="1" testid ="7600"> 
				///		<ExpectedInput>Invalid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///</TestCases>
				///<Update></Update>
				///<ArchivePlan></ArchivePlan>
				///<Interfaces></Interfaces>
				///
				/// <summary>
				/// This is not implemented.
				/// </summary>
				public virtual void Reset() // resets the enumerator
				{
					throw new BOL.BusinessObjectException("This method has not been implemented!");
				}                 

				///<Developers>
				///	<Developer>Greg Lohse</Developer>
				///</Developers>
				///<SiteName>Hines OIFO</SiteName>
				///<CreationDate>8/13/2005</CreationDate>
				///<TestCases>
				///	
				///<Case type="0" testid ="7601"> 
				///		<ExpectedInput>Valid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///<Case type="1" testid ="7602"> 
				///		<ExpectedInput>Invalid Parameters</ExpectedInput>
				///		<ExpectedOutput>NA</ExpectedOutput>
				///	</Case>
				///
				///</TestCases>
				///<Update></Update>
				///<ArchivePlan></ArchivePlan>
				///<Interfaces></Interfaces>
				///
				/// <summary>
				/// This returns the current object.
				/// </summary>
				public virtual Object Current 
				{
					get 
					{
						if (currentElement == _issuedUnits._issuedUnits)
						{
							if (_index == 0)
								throw new InvalidOperationException("Invalid Operation");
							else
								throw new InvalidOperationException("Invalid operation");
						}
						return currentElement;
					}
				}
			}

			#endregion
		}
	}
}
